# This module includes the Residue class and associated methods.
#
# Written by YW Huang <whuang@lineartime.com>
# 

"""This module provides a class to encapsulate a single residue
from an amino acid sequence, with option to build a backbone
using a Topology instance and all rotamers using a RotamerLib
instance.

Example:

  >>>from Residue import Residue;
  >>>need an example
"""

from Vector import Vector;
from math import *;

import Energy;
import Structure;
import Topology;
import copy;
import re;
import sys;


#
# aaMap constant
#
aaMap = {
                'A': 'ALA',
                'C': 'CYS',
                'D': 'ASP',
                'E': 'GLU',
                'F': 'PHE',
                'G': 'GLY',
                'H': 'HIS',
                'I': 'ILE',
                'K': 'LYS',
                'L': 'LEU',
                'M': 'MET',
                'N': 'ASN',
                'P': 'PRO',
                'Q': 'GLN',
                'R': 'ARG',
                'S': 'SER',
                'T': 'THR',
                'V': 'VAL',
                'W': 'TRP',
                'Y': 'TYR',
                'ALA': 'A',
                'CYS': 'C',
                'ASP': 'D',
                'GLU': 'E',
                'PHE': 'F',
                'GLY': 'G',
                'HIS': 'H',
                'ILE': 'I',
                'LYS': 'K',
                'LEU': 'L',
                'MET': 'M',
                'ASN': 'N',
                'PRO': 'P',
                'GLN': 'Q',
                'ARG': 'R',
                'SER': 'S',
                'THR': 'T',
                'VAL': 'V',
                'TRP': 'W',
                'TYR': 'Y'
        };



class Rotamer:
    """ This class encapsulates one rotamer for a residue.  It
        implements the following methods and attributes:

		__init__( residue, name, backbone )
		getName()
		setName( 'name' )
                getResidue()
		setBackbone( backbone )
		build( Topology, RotamerLib, residue, which = int )
		isBuilt() => True|False
		getRotamerAngles()
		getAtoms()
                chainEnergy(atomList, energyFunc, lj = None)
                residue
		name
		_isBuilt => True|False
		_atoms => { 'name': Atom, ... 'name': Atom }
		_chi => [ ang1, ... ang4 ]
    """

    def __init__( self, residue, name, backbone, top = None, rotolib = None, res = None, which = None ):
        self.residue = residue;
        self.name = name;
        self._atoms = copy.copy( backbone );
        self._isBuilt = False;
        self._chi = [];
        if ( not( (top == None) or (rotolib == None) or (res == None) or (which == None) ) ):
            self.build( top, rotolib, res, which );

    def chainEnergy( self, atomList, energyFunc, lj = None):
        """ Computes the energy between this rotamer and another list of atoms
            using the specified energy function.
        """

        if ( not self._isBuilt ):
            print "Warning: Rotamer", self.name, "of", self.residue, " has no sidechain.  Computing energy using backbone only!";

        return Energy.chainEnergy( self.getAtoms(True), atomList, energyFunc, lj );

    def getResidue( self ):
        return self.residue;

    def getName( self ):
        return self.name;

    def setName( self, name ):
        self.name = name;

    def setBackbone( self, backbone ):
        self._atoms = copy.copy( backbone );
	self._isBuilt = False;

    def build( self, top, rotolib, res, which ):
        buildNum = 0;
        for ic in top.findIcTargets( res, True ):
            if ( (not ic['target'][0] == '+') and (not self._atoms.has_key(ic['target'])) ):
                v1 = self._atoms[ic['v1Name']]['pos'];
                v2 = self._atoms[ic['v2Name']]['pos'];
                v3 = self._atoms[ic['v3Name']]['pos'];
                len34 = ic['len34'];
                ang234 = ic['ang234'];
                dih1234 = ic['dih1234'];
                if ( rotolib.hasResidue(res)  and  ic['rotatable']  and  not ic['improper'] ):
                    dih1234 = rotolib.getDihedral( res, which, ic['target'] );
                    if ( dih1234 == None ):
                        print >>sys.stderr, "rotatable proper bond for", res, ic['target'], "did not have a valid dihedral in the rotomer library at/for", which;
                        sys.exit(1);

                v4 = Structure.setVectorFromDihedral( v1, v2, v3, len34, ang234, dih1234 );
                self._atoms[ic['target']] = {
						'buildNum': buildNum,
						'pos': v4,
						'charge': top.atomCharge(res, ic['target'])
                                            };

            buildNum = buildNum + 1;

        self._isBuilt = True;
        if ( rotolib.hasResidue( res ) ):
            self._chi = rotolib.getRotamers( res, which )['chi'];

    def isBuilt( self ):
        return self._isBuilt;

    def getRotamerAngles( self ):
        if ( self._isBuilt ):
            return self._chi;

        return None;

    def getAtoms( self, sortedArray = False ):
        if ( not sortedArray ):
            return self._atoms;
        else:
            atomNames = self._atoms.keys();
            atomNames.sort( lambda x, y: self._atoms[x]['buildNum'] - self._atoms[y]['buildNum'] );
            ret = [];
            for atomName in atomNames:
                atom = copy.copy( self._atoms[atomName] );
                atom['name'] = atomName;
                ret.append( atom );

            return ret;


class Residue:
    """ This class is a container for a single residue from an amino
        acid sequence and implements methods/attributes:

		__init__( aa = None )
		getAA()
		getName()
		setAA( 'M' )
                setFromPDBAtoms( Topology, residue, atoms )
		buildBackbone( Topology, previousResidue = None )
		hasBackbone() => True|False
		hasRotamer() => True|False
		buildAnonymousRotamers( Topology, RotamerLib, which = None|int|list of int )
		buildRotamers( Topology, RotamerLib, which = None|int|list of int )
		numRotamers()
		resetRotamers()
		getBackboneAtoms()
                getRotamers()
                getRotamer( which = int )
		getRotamerAtoms( which = int|list of int )
		getSidechainAtoms( which = int|list of int )
		selectRotamer( which = int|name )
                selectedRotamer()
		getTorsionAngles( which = None|'phi'|'psi'|'omega' )
		getRotamerAngles( which = int|list of int )
		aa
		name
		_hasBackbone => True|False
		_hasRotamer => True|False
		_selectedRotamer => int (index)
		torsionAngles = { 'phi': ... 'psi': ... 'omega': ... }
		_rotamers = { num: Rotamer, ... num: Rotamer }
		backbone = { 'name': Atom, ... 'name': Atom }
    """

    def __init__( self, aa = None ):
        if ( not( aa == None ) and ( aaMap.has_key(aa) ) ):
            self.aa = aa;
            self.name = aaMap[aa];
        else:
            self.aa = None;
            self.name = None;

        self._hasBackbone = self._hasRotamer = False;
        self._selectedRotamer = None;
        self.torsionAngles = {};
        self._rotamers = [];
        self.backbone = {};


    def setFromPDBAtoms( self, top, res, atoms ):
        # set defaults
        self.name = res;
        self.aa = aaMap[res];
        self._hasBackbone = self._hasRotamer = True;
        self._selectedRotamer = 0;
        self.torsionAngles = top.getTorsionAngles( res );
        self._rotamers = [];
        self.backbone = {};

        # build backbone
        for bbAtom in Topology.backboneAtoms:
            if ( atoms.has_key(bbAtom) ):
                self.backbone[bbAtom] = atoms[bbAtom];

        # build rotamer
        rotName = res + '0000';
        rotamer = Rotamer( res, rotName, atoms );
        rotamer._isBuilt = True;
        rotamer._chi = Structure.getChiAngles( top, res, atoms );
        self._rotamers.append( rotamer );


    def getRotamer( self, which ):
        if ( which < len( self._rotamers ) ):
            return self._rotamers[which];
        else:
            return None;

    def getRotamers( self ):
        return self._rotamers;


    def selectedRotamer( self ):
        return self._selectedRotamer;


    def getAA( self ):
        return self.aa;


    def getName( self ):
        return self.name;


    def setAA( self, aa ):
        self.__init__( aa );


    def buildBackbone( self, top, prevRes = None ):
        res = self.name;
        if ( res == None ):
            # All backbones are the same regardless of residue
            res = "MET";

        if ( not self._hasBackbone ):
            buildNum = 0;
            if ( prevRes == None ):
                # special case for first residue
                self.backbone['N'] = {
					'buildNum': buildNum,
					'pos': Vector( 0, 0, 0 ),
					'charge': top.atomCharge(res, 'N')
                                     };
		buildNum = buildNum + 1;

                ic = top.findIcTarget( res, "N", 1 );
                self.backbone['CA'] = {
					'buildNum': buildNum,
					'pos': Vector( ic['len12'], 0, 0 ),
					'charge': top.atomCharge(res, 'CA')
                                      };
                buildNum = buildNum + 1;

                ic = top.findIcTarget( res, "C" );
                self.torsionAngles['phi'] = ic['dih1234'];
                x = self.backbone['CA']['pos'].x() + ( ic['len34'] * cos( radians( 180.0 - ic['ang234'] ) ) );
                y = self.backbone['CA']['pos'].y() + ( ic['len34'] * sin( radians( 180.0 - ic['ang234'] ) ) );
                self.backbone['C'] = {
					'buildNum': buildNum,
					'pos': Vector( x, y, 0 ),
					'charge': top.atomCharge(res, 'C')
                                     };
                buildNum = buildNum + 1;

                # build a virtual -C for placement of HN atom
                v1 = Structure.setVectorFromDihedral(
						self.backbone['C']['pos'],
						self.backbone['CA']['pos'],
						self.backbone['N']['pos'],
						ic['len12'],
						ic['ang123'],
						ic['dih1234']
                                                    );

                if ( not (res == 'PRO') ):
                    ic = top.findIcTarget( res, "HN" );
                    self.backbone['HN'] = { 'buildNum': buildNum };
                    self.backbone['HN']['pos'] = Structure.setVectorFromDihedral(
							v1,
							self.backbone[ic['v2Name']]['pos'],
							self.backbone[ic['v3Name']]['pos'],
							ic['len34'],
							ic['ang234'],
							ic['dih1234']
                                                                                );
                    self.backbone['HN']['charge'] = top.atomCharge(res, 'HN');
                    buildNum = buildNum + 1;

            else:
                # we need to build N and CA using the prev residue
                prevBackbone = prevRes.getBackboneAtoms();

                ic = top.findIcTarget( prevRes.getName(), "+N" );
                self.backbone['N'] = { 'buildNum': buildNum };
                self.backbone['N']['pos'] = Structure.setVectorFromDihedral(
						prevBackbone[ic['v1Name']]['pos'],
						prevBackbone[ic['v2Name']]['pos'],
						prevBackbone[ic['v3Name']]['pos'],
						ic['len34'],
						ic['ang234'],
						ic['dih1234']
                                                                      );
                self.backbone['N']['charge'] = top.atomCharge(res, 'N');
                buildNum = buildNum + 1;

                ic = top.findIcTarget( prevRes.getName(), "+CA" );
                self.backbone['CA'] = { 'buildNum': buildNum };
                self.backbone['CA']['pos'] = Structure.setVectorFromDihedral(
						prevBackbone[ic['v1Name']]['pos'],
						prevBackbone[ic['v2Name']]['pos'],
						self.backbone[ic['v3Name'][1:]]['pos'],
						ic['len34'],
						ic['ang234'],
						ic['dih1234']
                                                                       );
                self.backbone['CA']['charge'] = top.atomCharge(res, 'CA');
                buildNum = buildNum + 1;

                ic = top.findIcTarget( res, "C" );
                self.torsionAngles['phi'] = ic['dih1234'];
                self.backbone['C'] = { 'buildNum': buildNum };
                self.backbone['C']['pos'] = Structure.setVectorFromDihedral(
						prevBackbone[ic['v1Name'][1:]]['pos'],
						self.backbone[ic['v2Name']]['pos'],
						self.backbone[ic['v3Name']]['pos'],
						ic['len34'],
						ic['ang234'],
						ic['dih1234']
                                                                      );
                self.backbone['C']['charge'] = top.atomCharge(res, 'C');
                buildNum = buildNum + 1;

                if ( not (res == 'PRO') ):
                    ic = top.findIcTarget( res, "HN" );
                    self.backbone['HN'] = { 'buildNum': buildNum };
                    self.backbone['HN']['pos'] = Structure.setVectorFromDihedral(
							prevBackbone[ic['v1Name'][1:]]['pos'],
							self.backbone[ic['v2Name']]['pos'],
							self.backbone[ic['v3Name']]['pos'],
							ic['len34'],
							ic['ang234'],
							ic['dih1234']
                                                                                );
                    self.backbone['HN']['charge'] = top.atomCharge(res, 'HN');
                    buildNum = buildNum + 1;


            # get omega torsion angle
            ic = top.findIcTarget( res, "+CA" );
            self.torsionAngles['omega'] = ic['dih1234'];

            ic = top.findIcTarget( res, "+N" );
            self.torsionAngles['psi'] = ic['dih1234'];
            v1 = Structure.setVectorFromDihedral(
						self.backbone['N']['pos'],
						self.backbone['CA']['pos'],
						self.backbone['C']['pos'],
						ic['len34'],
						ic['ang234'],
						ic['dih1234']
                                                );
            ic = top.findIcTarget( res, "O" );
            self.backbone['O'] = { 'buildNum': buildNum };
            self.backbone['O']['pos'] = Structure.setVectorFromDihedral(
						v1,
						self.backbone['CA']['pos'],
						self.backbone['C']['pos'],
						ic['len34'],
						ic['ang234'],
						ic['dih1234']
                                                                  );
            self.backbone['O']['charge'] = top.atomCharge(res, 'O');
            buildNum = buildNum + 1;

            # build backbone atoms (that haven't already been built)
            for ic in top.findIcBackbone( res ):
                if ( ic  and  not self.backbone.has_key( ic['target'] ) ):
                    v1 = self.backbone[ic['v1Name']]['pos'];
                    v2 = self.backbone[ic['v2Name']]['pos'];
                    v3 = self.backbone[ic['v3Name']]['pos'];
                    v4 = Structure.setVectorFromDihedral(
						v1,
						v2,
						v3,
						ic['len34'],
						ic['ang234'],
						ic['dih1234']
                                                        );
                    self.backbone[ic['target']] = {
						'buildNum': buildNum,
						'pos': v4,
						'charge': top.atomCharge(res, ic['target'])
                                                  };
                    buildNum = buildNum + 1;

            self._hasBackbone = True;


    def hasBackbone( self ):
        return self._hasBackbone;


    def hasRotamer( self ):
        return self._hasRotamer;

    def resetRotamers( self ):
        self._hasRotamer = False;
        self._rotamers = [];

    def buildAnonymousRotamers( self, top, rotolib, which = None ):
        """ This method will build the selected rotamers for an
            anonymous residue.  In this case, which is either a single
            residue code (i.e. 'MET'), a list of residue codes, or
            None.  These correspond to building all rotamers for the
            single selected residue, list of residues, or all residues
            respectively.
        """ 
        if ( self._hasBackbone ):
            if ( self.name == None ):
                resList = [];
                if ( which == None ):
                    for res in aaMap.keys():
                        if ( len(res) == 3 ):
                            resList.append(res);
                elif ( type(which) == str ):
                    resList.append(which);
                elif ( type(which) == list ):
                    resList = which;

                for res in resList:
                    # build all rotamers for the selected residues
                    if ( rotolib.hasResidue( res ) ):
                        rlEnts = rotolib.getRotamers( res );
                        for idx, rlEnt in enumerate(rlEnts):
                            rotamer = Rotamer( res, rlEnt['name'], self.backbone );
                            rotamer.build( top, rotolib, res, idx );
                            self._rotamers.append( rotamer );
                    else:
                        # no rotamer library entries so just building only possible
                        rotamerName = res + "0000";
                        rotamer = Rotamer( res, rotamerName, self.backbone );
                        rotamer.build( top, rotolib, res, None );
                        self._rotamers.append( rotamer );
                        
                self._hasRotamer = True;
            else:
                print "Error: Can't build anonymous rotamers for a known residue.";

    def numRotamers( self ):
        """ Returns the number of rotamers built for this residue.
        """
        return len( self._rotamers );

    def buildRotamers( self, top, rotolib, which = None ):
        """ This method will build the selected rotamers for this residue.
            If the residue is unknown (anonymous) or unset, the
            buildAnonymousRotamers method is called instead.  which
            specifies whether to build a single rotamer (integer index
            value), a list of rotamers (list of integers), or all
            rotamers (which == None).  In the case of an anonymous
            residue, which can contain the name of a single residue,
            a list of residues, or None to build all rotamers for the
            selected residue, list of residues, or all residues,
            respectively.
        """
        if ( self._hasBackbone ):
            if ( self.name == None ):
                self.buildAnonymousRotamers( top, rotolib, which );
            elif ( rotolib.hasResidue( self.name ) ):
                numRotamers = rotolib.numRotamers( self.name );
                if ( (type(which) == int)  and  (which < numRotamers) ):
                    rlEnt = rotolib.getRotamers( self.name, which );
                    rotamer = Rotamer( self.name, rlEnt['name'], self.backbone );
                    rotamer.build( top, rotolib, self.name, which );
                    self._rotamers.append( rotamer );
                    self._hasRotamer = True;
                elif ( which == None ):
                    rlEnts = rotolib.getRotamers( self.name, which );
                    for idx, rlEnt in enumerate(rlEnts):
                        rotamer = Rotamer( self.name, rlEnt['name'], self.backbone );
                        rotamer.build( top, rotolib, self.name, idx );
                        self._rotamers.append( rotamer );

                    self._hasRotamer = True;
                elif ( type(which) == list ):
                    rlEnts = rotolib.getRotamers( self.name, which );
                    for idx, rlEnt in rlEnts.iteritems():
                        rotamer = Rotamer( self.name, rlEnt['name'], self.backbone );
                        rotamer.build( top, rotolib, self.name, idx );
                        self._rotamers.append( rotamer );

                    self._hasRotamer = True;
            else:
                # no rotamer library entries so just building only possible
                rotamerName = self.name + "0000";
                rotamer = Rotamer( self.name, rotamerName, self.backbone );
                rotamer.build( top, rotolib, self.name, None );
                self._rotamers.append( rotamer );
                self._hasRotamer = True;


    def getBackboneAtoms( self, sortedArray = False ):
        """ Return a data structure containing the atoms for this residue's
            backbone.  This is stored (and thus returned) natively as a 
            hash with the key being the unique atom name.  However,
            specify sortedArray == True will return an array of atoms
            sorted by their build order.
        """
        if ( self._hasBackbone ):
            if ( not sortedArray):
                return self.backbone;
            else:
                atomNames = self.backbone.keys();
                atomNames.sort( lambda x, y: self.backbone[x]['buildNum'] - self.backbone[y]['buildNum'] );
                ret = [];
                for atomName in atomNames:
                    atom = copy.copy( self.backbone[atomName] );
                    atom['name'] = atomName;
                    ret.append( atom );

                return ret;

        return None;


    def getRotamerAtoms( self, which, sortedArray = False ):
        """ Get rotamer (and backbone) atoms for the specified rotamer(s)
            associated with this residue.  By default, a hash is returned
            for each rotamer using the unique atom name as the key,
            however passing sortedArray == True will return an array
            of atoms sorted by build order.
        """
        if ( not self._hasRotamer ):
            return None;
        elif ( (type(which) == int)  and  (which < len(self._rotamers)) ):
            return self._rotamers[which].getAtoms(sortedArray);
        elif ( type(which) == list ):
            ret = {};
            for idx in which:
                if ( idx < len( self._rotamers ) ):
                    ret[idx] = self._rotamers[idx].getAtoms(sortedArray);
                else:
                    ret[idx] = None;

            return ret;
        else:
            return None;
        
    def getSidechainAtoms( self, which = None, sortedArray = False ):
	"""Similar to getRotamerAtoms, but returns only atoms in the
	   sidechain itself.  Uses getRotamerAtoms to retrieve a list from
	   which backbone atomsw are filtered.  By default, a hash is
	   returned for each rotamer using the unique atom name as the key,
	   however passing sortedArray == True will return an array
	   of atoms sorted by build order.
        """
	if which == None:
	    which = self._selectedRotamer;
	return [x for x in self.getRotamerAtoms(which,sortedArray) if x['name'] not in self.backbone.keys()];

    def selectRotamer( self, which ):
        if ( which < len(self._rotamers) ):
            self._selectedRotamer = which;
            self.name = self._rotamers[which].getResidue();
            self.aa = aaMap[self.name];


    def getTorsionAngles( self, which = None ):
        if ( not self._hasBackbone ):
            return None;
        elif ( which == None ):
            return self.torsionAngles;
        elif ( self.torsionAngles.has_key(which) ):
            return self.torsionAngles[which];
        else:
            return None;


    def getRotamerAngles( self, which ):
        if ( not self._hasRotamer ):
            return None;
        elif ( (type(which) == int)  and  (which < len(self._rotamers)) ):
            return self._rotamers[which].getRotamerAngles();
        elif ( type(which) == list ):
            ret = {};
            for idx in which:
                if ( idx < len(self._rotamers) ):
                    ret[idx] = self._rotamers[idx].getRotamerAngles();
                else:
                    ret[idx] = None;

            return ret;
        else:
            return None;
