# This module contains the Protein class definition and methods
#
# Written by YW Huang <whuang@lineartime.com>
# 

"""This module provides a class that represents a protein (ordered
collection of residues) built from an amino acid sequence.

Example:

  >>>from Protein import Protein;
  >>>need examples
"""

import Residue;
import re;
import sys;

from Vector import Vector;

class Protein:
    """ This class is a container for a protein.  It contains an
        ordered collection of Residue objects and implements the
        following attributes and methods:

		__init__( self, top, rotolib, sequence = None, buildRotamers = False )
                getSequence( self )
                setSequence( self, sequence )
                getResidues( self, idx = None )
                initializeResidues( self )
                toPDB( self, fileName = None )
                _residues = [ Residue, ... ]
                _sequence = "MRRTV..."
                _top = ref(Topology)
                _rotolib = ref(RotamerLib)
                _seqLen = len(_sequence)
                _autoBuild = False|True
    """

    def __init__( self, top, rotolib, sequence = None, buildRotamers = False ):
        self._sequence = sequence;
        self._residues = [];
        self._top = top;
        self._rotolib = rotolib;
        self._autoBuild = buildRotamers;
        if ( sequence ):
            self._seqLen = len(sequence);
            self.initializeResidues( buildRotamers );
        else:
            self._seqLen = 0;

    def loadFromPDB( self, top, file ):
        self._sequence = "";
        self._residues = [];
        self._autoBuild = False;
        self._seqLen = 0;

        inp = open( file, 'r' );
        if ( inp ):
            atomline = re.compile('^ATOM\s+');
            curResNum = -1;
            curRes = None;
            residueAtoms = {};

            for line in inp.readlines():
                line = line.strip();
                if ( atomline.match(line) ):
                    buildNum = int( line[6:11] );
                    atomName = line[12:16].strip();
                    resName = line[17:21].strip();
                    thisResNum = int( line[22:26] );
                    x = float( line[30:38] );
                    y = float( line[38:46] );
                    z = float( line[46:54] );

                    atom = {
				'buildNum': buildNum,
				'pos': Vector(x, y, z),
				'charge': top.atomCharge(resName, atomName)
			   };

                    if ( curResNum == thisResNum ):
                        # atom belongs to current residue, so add it
                        if ( residueAtoms.has_key(atomName) ):
                            print "Warning: Residue", thisResNum, resName, "already has atom", atomName, "- skipping";
                        else:
                            residueAtoms[atomName] = atom;
                    else:
                        # atom belongs to next residue
                        if (len(residueAtoms) > 0):
                            # save the residue we were building
                            residue = Residue.Residue();
                            residue.setFromPDBAtoms( top, curRes, residueAtoms );
                            self._residues.append(residue);
                            self._sequence = self._sequence + Residue.aaMap[curRes];
                            self._seqLen = self._seqLen + 1;

                        # reset
                        curResNum = thisResNum;
                        curRes = resName;
                        residueAtoms = { atomName: atom };

            inp.close();

            # build last residue
            residue = Residue.Residue();
            residue.setFromPDBAtoms( top, curRes, residueAtoms );
            self._residues.append(residue);
            self._sequence = self._sequence + Residue.aaMap[curRes];
            self._seqLen = self._seqLen + 1;


    def getSequence( self ):
        return self._sequence;


    def setSequence( self, sequence ):
        self.__init__( self._top, self._rotolib, sequence, self._autoBuild );


    def getResidues( self, idx = None ):
        if ( (type(idx) == int)  and  (idx < len(self._residues)) ):
            return self._residues[idx];
        elif ( idx == None ):
            return self._residues;
        else:
            return None;


    def initializeResidues( self, buildRotamers = False ):
        for i in range(self._seqLen):
            aa = self._sequence[i].upper();
            r = Residue.Residue(aa);
            if ( i == 0 ):
                r.buildBackbone( self._top );
            else:
                r.buildBackbone( self._top, self._residues[-1] );

            if ( buildRotamers ):
                r.buildRotamers( self._top, self._rotolib );

            self._residues.append(r);


    def toPDB( self, fileName = None, buildRotamers = False ):
        outp = sys.stdout;
        if ( not( fileName == None ) ):
            outp = open( fileName, "w" );
            if ( not outp ):
                print >>sys.stderr, "could not obtain write filehandle for", fileName;
                sys.exit(1);

        atomNum = 1;
        for resIdx, res in enumerate( self._residues ):
            resNum = resIdx + 1;
            if ( not res.hasRotamer() ):
                if ( buildRotamers ):
                    res.buildRotamers( self._top, self._rotolib );
                else:
                    print >>sys.stderr, "residue", resNum, res.getName(), "rotamers have not yet been built."
                    sys.exit(1);

            selRotamer = None;
            if ( res.selectedRotamer() == None ):
                selRotamer = res._rotamers.keys()[0];
            else:
                selRotamer = res.selectedRotamer();

            atoms = res.getRotamerAtoms( selRotamer, True );
            if ( atoms == None ):
                print >>sys.stderr, "residue", resNum, res.getName(), "rotamer", selRotamer, "has no atoms!";
                sys.exit(1);

            for atom in atoms:
                print >>outp, "%-6s%5d %-4s %-4s %4d    %8.3f%8.3f%8.3f%6.2f%6.2f      %-4s%2s%2s" % ("ATOM", atomNum, atom['name'], res.getName(), resNum, atom['pos'].x(), atom['pos'].y(), atom['pos'].z(), 0.0, 0.0, "", atom['name'][0], "");
                atomNum = atomNum + 1;

        if ( not( fileName == None ) ):
            outp.close();
