function aModel = InitializeModelRun(aModel, RunNumber, Name, varargin)
% 
%**************************************************************************
%
% $Id: InitializeModelRun.m,v 1.3 2004/11/15 18:30:15 bhendrik Exp $
% $Revision: 1.3 $
% $Author: bhendrik $
% $Date: 2004/11/15 18:30:15 $
% Type: Matlab script.
%
% Comments: This function initializes a ModelRun, number 'RunNumber' so 
% that the Model can be sent to SolveModel 
%
% Written by Bart Hendriks
%
%**************************************************************************
%
% The specification of individual IC or PV takes precedence over IC & PV
% specified in files which takes precedence over the default IC & PV (found
% in Model.Species.IC and Model.Parameter.Value).
%
% Allowable syntax:
% aModel = InitializeModelRun(aModel, RunNumber, Name, ICfilename, PVfilename, property1, value1, property2, value2, ...)
% aModel = InitializeModelRun(aModel, RunNumber, Name, RefModelRun, property1, value1, property2, value2, ...)
% aModel = InitializeModelRun(aModel, RunNumber, Name, property1, value1, property2, value2, ...)
%
% ICfilename = .m file with struct Species with field 'IC'
% PVfilename = .m file with struct Parameter with field 'Value'
% RefModelRun = the number of the model run from which you want to import
% the final values; 
%
% Default IC and PV can be obtained by having ICfilename = ''; and
% PVfilename = '';  (empty string) If they are [] you will get problems.
%
% values need to be specifies in nM or (#/cell)/PMCF*CPL, for example.
%


if size(aModel.ModelRun,2) == 0
    aModel.ModelRun = ModelRun;
else
    if RunNumber > size(aModel.ModelRun,2)
        aModel.ModelRun(RunNumber) = ModelRun;
    end;
end;

aModel.ModelRun(RunNumber).Name = Name;

% this pulls out the conversion factors so that the input files can also
% have things entered as #/cell.
CPL = aModel.ConversionFactor.CPL;
PMCF = aModel.ConversionFactor.PMCF;
EMCF = aModel.ConversionFactor.EMCF;
CYCF = aModel.ConversionFactor.CYCF;
NUCF = aModel.ConversionFactor.NUCF;



% this piece figures out if there are IC and PV files in the input
% the possibilities are: 
% 'ICfilename', [],
% 'ICfilename', '',
% [], 'PVfilename,
% '', 'PVfilename',
% 'ICfilename, 'PVfilename'
FileInput = 0;
if isa(varargin{1}, 'char') & isa(varargin{2}, 'char')
    FileInput = 1;
elseif isa(varargin{1}, 'char') & isempty(varargin{2})
    varargin{2} = '';
    FileInput = 1;
elseif isa(varargin{2}, 'char') & isempty(varargin{1})
    varargin{1} = '';
    FileInput = 1;
end;
if FileInput ==1
    ICfilename = varargin{1};
    PVfilename = varargin{2};
    property_argin = varargin(3:end);
else
    ICfilename = '';
    PVfilename = '';
    property_argin = varargin;
end;



if isa(varargin{1}, 'double')
    % in this case IC & PV are going to come from the final values from run # 'RefModelRun' 
    RefModelRun = varargin{1};
    if RefModelRun > size(aModel.ModelRun,2)
        error('Reference ModelRun must be an existing ModelRun');
    else
        if ~isempty(aModel.ModelRun(RefModelRun).Results)
            for i=1:size(aModel.Species,2)
                aModel.ModelRun(RunNumber).SpeciesIC(i) = aModel.ModelRun(RefModelRun).Results(end, i);
            end;
            for i=1:size(aModel.Parameter,2)
                aModel.ModelRun(RunNumber).ParameterValues(i) = aModel.ModelRun(RefModelRun).ParameterValues(i);
            end;
            for i=1:size(aModel.Compartment,2)
                aModel.ModelRun(RunNumber).CompartmentVolumes(i) = aModel.ModelRun(RefModelRun).CompartmentVolumes(i);
            end;
        else
            error('No results present in reference model run');
        end;
    end;
    property_argin = varargin(2:end);
else
    % 
    % 
    % IC & PV may come from files or default parameters
    if ~isempty(ICfilename)
        if strcmp(ICfilename(end-1:end), '.m')
            ICfilename = ICfilename(1:end-2);
        end;
    end;
    run(ICfilename);
    %ICfilename MUST contain a struct called Species with field IC!
    if ~exist('Species', 'var')
        Species = struct('IC', {});
    end;


    for i=1:size(aModel.Species,2)
        if i <= size(Species,2)
            if ~isempty(Species(i).IC)
                Species(i).IC;
                aModel.ModelRun(RunNumber).SpeciesIC(i) = Species(i).IC;
            else
                aModel.ModelRun(RunNumber).SpeciesIC(i) = aModel.Species(i).IC;
            end;
        else
            if ~isempty(aModel.Species(i).IC)
                aModel.ModelRun(RunNumber).SpeciesIC(i) = aModel.Species(i).IC;
            else
                aModel.ModelRun(RunNumber).SpeciesIC(i) = 0;
                fprintf('\n\nCAUTION! Species %g is empty\n\n', i);
            end;
        end;
    end;


    if ~isempty(PVfilename)
        if strcmp(PVfilename(end-1:end), '.m')
            PVfilename = PVfilename(1:end-2);
        end;
    end;
    run(PVfilename);
    %PVfilename MUST contain a struct called Species with field IC!
    if ~exist('Parameter', 'var')
        Parameter = struct('PV', {});
    end;
    for i=1:size(aModel.Parameter,2)
        if i <= size(Parameter,2)
            if ~isempty(Parameter(i).Value)
                Parameter(i).Value;
                aModel.ModelRun(RunNumber).ParameterValues(i) = Parameter(i).Value;
            else
                aModel.ModelRun(RunNumber).ParameterValues(i) = aModel.Parameter(i).Value;
            end;
        else
            if ~isempty(aModel.Parameter(i).Value)
                aModel.ModelRun(RunNumber).ParameterValues(i) = aModel.Parameter(i).Value;
            else
                aModel.ModelRun(RunNumber).ParameterValues(i) = 0;
                fprintf('\n\nCAUTION: Parameter %g is empty\n\n', i);
            end;
        end;
    end;

    if ~exist('Compartment', 'var')
        Compartment = struct('Size', {});
    end;
    for i=1:size(aModel.Compartment,2)
        if i <= size(Compartment,2)
            if ~isempty(Compartment(i).Size)
                Compartment(i).Value;
                aModel.ModelRun(RunNumber).CompartmentVolumes(i) = Compartment(i).Size;
            else
                aModel.Compartment(i).Size;
                aModel.ModelRun(RunNumber).CompartmentVolumes(i) = aModel.Compartment(i).Size;
            end;
        else
            aModel.ModelRun(RunNumber).CompartmentVolumes(i) = aModel.Compartment(i).Size;
        end;
    end;
     
end;


% Now replace individual IC and Parameters as specified:
while length(property_argin) >= 2,
    prop = property_argin{1};
    val = property_argin{2};
    property_argin = property_argin(3:end);
    
    % determine if string is a Species or Parameter (or both)
    warning off;
    FoundIt = 0;
    if GPN(aModel.Species, prop) ~= 0 
        index = GPN(aModel.Species, prop);
        aModel.ModelRun(RunNumber).SpeciesIC(index) = val;
        FoundIt = 1;
    end;
    if GPN(aModel.Parameter, prop) ~= 0
        index = GPN(aModel.Parameter, prop);
        aModel.ModelRun(RunNumber).ParameterValues(index) = val;
        FoundIt = 1;
    end;
    if GPN(aModel.Compartment, prop) ~= 0 
        index = GPN(aModel.Compartment, prop);
        aModel.ModelRun(RunNumber).CompartmentVolumes(index) = val;
        FoundIt = 1;
    end;
    
    if GPN(aModel.Species, prop) ~= 0 & GPN(aModel.Parameter, prop) ~= 0
        warning on;
        warning('Species & Parameter with same name!');
    end;
    warning on;
    if FoundIt == 0
        warning('Could not find %s in Species, Parameter or Compartment lists.', prop);
    end;
    
end;

return;