function aModel = GenerateODEfile(aModel, format)
% This file takes a Model and writes the corresponding ODEfile
%**************************************************************************
%
% $Id: GenerateODEfile.m,v 1.3 2004/11/15 18:29:47 bhendrik Exp $
% $Revision: 1.3 $
% $Author: bhendrik $
% $Date: 2004/11/15 18:29:47 $
% Type: Matlab script.
%
% Comments: 
%
% Written by Bart Hendriks
%
%**************************************************************************
% Bart Hendriks
% 26-Aug-2004
% This file takes a bunch of inputs and writes a
% corresponding ODEFile in a matlab-solvable format or in text for
% human-interpretation. 
%
%
% SMBL-esque format
% Each species is in concentration units. Every Flux expression in the
% reaction struct must be absolute fluxes (changes in absolute amounts of
% species, not changes in concentration).
% 1 : matlab odefile - 
% 2 : text odefile - 

if size(aModel,2) > 1
   error('Can''t write ODEfile for more than one model!');
   return;
end;

% Make sure the Model name doesn't have any evil special characters that
% would make for an invalid filename:
if isempty(aModel.Name) | ~isempty(strfind(aModel.Name, ' ')) | ...
        ~isempty(strfind(aModel.Name, '~')) | ~isempty(strfind(aModel.Name, '+')) | ...
        ~isempty(strfind(aModel.Name, '`')) | ~isempty(strfind(aModel.Name, '=')) | ...
        ~isempty(strfind(aModel.Name, '!')) | ~isempty(strfind(aModel.Name, '{')) | ...
        ~isempty(strfind(aModel.Name, '@')) | ~isempty(strfind(aModel.Name, '}')) | ...
        ~isempty(strfind(aModel.Name, '#')) | ~isempty(strfind(aModel.Name, '[')) | ...
        ~isempty(strfind(aModel.Name, '$')) | ~isempty(strfind(aModel.Name, ']')) | ...
        ~isempty(strfind(aModel.Name, '%')) | ~isempty(strfind(aModel.Name, ':')) | ...
        ~isempty(strfind(aModel.Name, '^')) | ~isempty(strfind(aModel.Name, ';')) | ...
        ~isempty(strfind(aModel.Name, '&')) | ~isempty(strfind(aModel.Name, '"')) | ...
        ~isempty(strfind(aModel.Name, '*')) | ~isempty(strfind(aModel.Name, '''')) | ...
        ~isempty(strfind(aModel.Name, '(')) | ~isempty(strfind(aModel.Name, '<')) | ...
        ~isempty(strfind(aModel.Name, ')')) | ~isempty(strfind(aModel.Name, '>')) | ...
        ~isempty(strfind(aModel.Name, '-')) | ~isempty(strfind(aModel.Name, ',')) | ...
        ~isempty(strfind(aModel.Name, '.')) | ~isempty(strfind(aModel.Name, '.')) | ...
        ~isempty(strfind(aModel.Name, '?')) 
    error('Can''t generate legal ODEfile name. Remove special characters from model name');
    return;
else
    filename = cat(2, aModel.Name, '_ODE.m');
    aModel = set(aModel, 'ODEfilename', filename);
end;

% Check to see if and ODEfile with the same name already exists:
if exist(filename, 'file')
    fprintf('\nWarning: The ODEfile %s already exists!\n', filename);
    reply = input('Do you want to overwrite it (y/n)? ', 's');
    if ~(strcmp(reply, 'y') | strcmp(reply, 'Y'))
        fprintf('Skipping ODEfile generation ...\n');
        return;
    else
        fprintf('Overwriting file ...\n');
    end;
end;


fid = fopen(filename,'w');

% format == 1 : generate odefile for matlab
% format == 2 : geneate text odefile for human interpreting


fprintf('\nWRITING ODE FILE: %s ...\n', filename);

% I use S, P, R, C to avoid confusion with the classes Species, Parameter
% Reaction, Compartment.
S = aModel.Species; % get the Species out of the model
P = aModel.Parameter; % get the Parameters out of the model
R = aModel.Reaction; % get the Reactions out of the model
C = aModel.Compartment; % get the Compartments out of the model

ODE = struct('eqn', [], 'usage', []);
    

%//////////////////////////////////////////////////////////////////////////
% This piece deals with the Compound Species (a.k.a Species that have
% sub-species
for i=1:size(S,2)
    
    ODE(i).eqn = '';
    ODE(i).usage = 0;
        
    % This little bit checks to see if the species number corresponds
    % on one of the compound species and, if so, adds the appropriate
    % expression(s) to the odefile.
    
    if ~isempty(S(i).SubSpecies)
        ODE(i).eqn = cat(2, ODE(i).eqn, '(');
        for k=1:size(S(i).SubSpecies,2)
            ODE(i).usage = ODE(i).usage + 1;
            switch format
                case 1
                    ODE(i).eqn = cat(2, ODE(i).eqn, sprintf(' + dydt(%g,1)*v(%g)',  GPN(S, S(i).SubSpecies{k}), GPN(C, S(GPN(S, S(i).SubSpecies{k})).Location) ));
                case 2
                    ODE(i).eqn = cat(2, ODE(i).eqn, sprintf(' + dydt(%s)*volume(%s)',  S(i).SubSpecies{k}, S(GPN(S, S(i).SubSpecies{k})).Location));
            end;
        end;  % Loop through Sub-Species
    
        switch format
            case 1
                ODE(i).eqn = cat(2, ODE(i).eqn, sprintf(')/v(%g)', GPN(C, S(i).Location)));
            case 2
                ODE(i).eqn = cat(2, ODE(i).eqn, sprintf(')/volume(%s)', S(i).Location));
        end;
    end; 
end; % loop through Species
%//////////////////////////////////////////////////////////////////////////





%//////////////////////////////////////////////////////////////////////////
% Now go through the entire list of reactions and add each reaction to the
% appropriate equation for each species after switching all the reactants, 
% modifiers and products with y(#) and the parameters with p(#), volumes 
% with v(#) and so on...

for j=1:size(R,2)
    
    % Get the number of each Reactant/Product/Parameter/Modifier to see if
    % it is involved:
    if ~isempty(R(j).Reactant_1) & ~isnan(R(j).Reactant_1)
        Reactant_1 = GPN(S, R(j).Reactant_1);
        R1_Volume = GPN(C, S(Reactant_1).Location);
    else
        Reactant_1 = 0;
        R1_Volume = 0;
    end;
    if ~isempty(R(j).Reactant_2) & ~isnan(R(j).Reactant_2)
        Reactant_2 = GPN(S, R(j).Reactant_2);
        R2_Volume = GPN(C, S(Reactant_2).Location);
    else
        Reactant_2 = 0;
        R2_Volume = 0;
    end;
    if ~isempty(R(j).Product_1) & ~isnan(R(j).Product_1)
        Product_1 = GPN(S, R(j).Product_1);
        P1_Volume = GPN(C, S(Product_1).Location);
    else
        Product_1 = 0;
        P1_Volume = 0;
    end;
    if ~isempty(R(j).Product_2) & ~isnan(R(j).Product_2)
        Product_2 = GPN(S, R(j).Product_2);
        P2_Volume = GPN(C, S(Product_2).Location);
    else 
        Product_2 = 0;
        P2_Volume = 0;
    end;
    
    if ~isempty(R(j).Modifier_1) & ~isnan(R(j).Modifier_1)
        Modifier_1 = GPN(S, R(j).Modifier_1);
        M1_Volume = GPN(C, S(Modifier_1).Location);
    else
        Modifier_1 = 0;
        M1_Volume = 0;
    end;
    if ~isempty(R(j).Modifier_2) & ~isnan(R(j).Modifier_2)
        Modifier_2 = GPN(S, R(j).Modifier_2);
        M2_Volume = GPN(C, S(Modifier_2).Location);
    else
        Modifier_2 = 0;
        M2_Volume = 0;
    end;
    if ~isempty(R(j).Modifier_3) & ~isnan(R(j).Modifier_3)
        Modifier_3 = GPN(S, R(j).Modifier_3);
        M3_Volume = GPN(C, S(Modifier_3).Location);
    else
        Modifier_3 = 0;
        M3_Volume = 0;
    end;
                
    if ~isempty(R(j).Parameter_1) & ~isnan(R(j).Parameter_1)
        Parameter_1 = GPN(P, R(j).Parameter_1);
    else
        Parameter_1 = 0;
    end;
    if ~isempty(R(j).Parameter_2) & ~isnan(R(j).Parameter_2)
        Parameter_2 = GPN(P, R(j).Parameter_2);
    else
        Parameter_2 = 0;
    end;
    if ~isempty(R(j).Parameter_3) & ~isnan(R(j).Parameter_3)
        Parameter_3 = GPN(P, R(j).Parameter_3);
    else
        Parameter_3 = 0;
    end;
    if ~isempty(R(j).Parameter_4) & ~isnan(R(j).Parameter_4)
        Parameter_4 = GPN(P, R(j).Parameter_4);
    else
        Parameter_4 = 0;
    end;
    if ~isempty(R(j).Parameter_5) & ~isnan(R(j).Parameter_5)
        Parameter_5 = GPN(P, R(j).Parameter_5);
    else
        Parameter_5 = 0;
    end;
    
    warning off;
    PM_Volume = GPN(C, 'Plasma Membrane');
    EM_Volume = GPN(C, 'Endosomal Membrane');
    EL_Volume = GPN(C, 'Endosomal Lumen');
    CY_Volume = GPN(C, 'Cytoplasm');
    Cell_Volume = GPN(C, 'Cell');
    SY_Volume = GPN(C, 'Synovium');
    warning on;
         
    Flux = R(j).Flux;
    if isempty(Flux)
        Flux = '';
    end;
    % all fluxes are converted into absolute amounts (by multiplying each Species
    % concentration by its compartment volume). The final output is converted back into
    % concentration by dividing by the product's compartment
    % volume. 
                
    % The flux can be an arbitrary expression containing
    % keywords: Reactant_1 Reactant_2 Product_1 Product_2
    % Modifier_1 _2 _3 and Parameter_1 _2 _3 _4 _5.
    
    
    % Now do lots of string replacements to switch keywords with y(#) and
    % so on...
                
    switch format
        case 1 
            Name = cat(2, 'y(', num2str(Reactant_1), ')');
            Flux = strrep(Flux, 'Reactant_1', Name);
            Name = cat(2, 'y(', num2str(Reactant_2), ')');
            Flux = strrep(Flux, 'Reactant_2', Name);
                        
            Name = cat(2, 'y(', num2str(Product_1), ')');
            Flux = strrep(Flux, 'Product_1', Name);
            Name = cat(2, 'y(', num2str(Product_2), ')');
            Flux = strrep(Flux, 'Product_2', Name);
                        
            % Special case ....................
            % This must preceed the other Modifier_1 strrep!!
            % 
            %  
            
            Name = cat(2, 'dydt(', num2str(Modifier_1));
            Flux = strrep(Flux, 'dydt(Modifier_1', Name);
            Flux = strrep(Flux, 'dydt( Modifier_1', Name);
                        
            Name = cat(2, 'dydt(', num2str(Modifier_2));
            Flux = strrep(Flux, 'dydt(Modifier_2', Name);
            Flux = strrep(Flux, 'dydt( Modifier_2', Name);
                        
            Name = cat(2, 'dydt(', num2str(Modifier_3));
            Flux = strrep(Flux, 'dydt(Modifier_3', Name);
            Flux = strrep(Flux, 'dydt( Modifier_3', Name);
                        
            % end .........................................
                        
            Name = cat(2, 'y(', num2str(Modifier_1), ')');
            Flux = strrep(Flux, 'Modifier_1', Name);
            Name = cat(2, 'y(', num2str(Modifier_2), ')');
            Flux = strrep(Flux, 'Modifier_2', Name);
            Name = cat(2, 'y(', num2str(Modifier_3), ')');
            Flux = strrep(Flux, 'Modifier_3', Name);
                
            Name = cat(2, 'p(', num2str(Parameter_1), ')');
            Flux = strrep(Flux, 'Parameter_1', Name);    
            Name = cat(2, 'p(', num2str(Parameter_2), ')');
            Flux = strrep(Flux, 'Parameter_2', Name);   
            Name = cat(2, 'p(', num2str(Parameter_3), ')');
            Flux = strrep(Flux, 'Parameter_3', Name);   
            Name = cat(2, 'p(', num2str(Parameter_4), ')');
            Flux = strrep(Flux, 'Parameter_4', Name);   
            Name = cat(2, 'p(', num2str(Parameter_5), ')');
            Flux = strrep(Flux, 'Parameter_5', Name);   
                        
            Name = cat(2, 'v(', num2str(R1_Volume), ')');
            Flux = strrep(Flux, 'R1_Volume', Name);
            Name = cat(2, 'v(', num2str(R2_Volume), ')');
            Flux = strrep(Flux, 'R2_Volume', Name);
            Name = cat(2, 'v(', num2str(P1_Volume), ')');
            Flux = strrep(Flux, 'P1_Volume', Name);
            Name = cat(2, 'v(', num2str(P2_Volume), ')');
            Flux = strrep(Flux, 'P2_Volume', Name);
            Name = cat(2, 'v(', num2str(M1_Volume), ')');
            Flux = strrep(Flux, 'M1_Volume', Name);
            Name = cat(2, 'v(', num2str(M2_Volume), ')');
            Flux = strrep(Flux, 'M2_Volume', Name);
            Name = cat(2, 'v(', num2str(M3_Volume), ')');
            Flux = strrep(Flux, 'M3_Volume', Name);
                        
            if PM_Volume > 0
                Name = cat(2, 'v(', num2str(PM_Volume), ')');
                Flux = strrep(Flux, 'PM_Volume', Name);
            end;
            if EM_Volume > 0
                Name = cat(2, 'v(', num2str(EM_Volume), ')');
                Flux = strrep(Flux, 'EM_Volume', Name);
            end;
            if EL_Volume > 0
                Name = cat(2, 'v(', num2str(EL_Volume), ')');
                Flux = strrep(Flux, 'EL_Volume', Name);
            end;
            if CY_Volume > 0
                Name = cat(2, 'v(', num2str(CY_Volume), ')');
                Flux = strrep(Flux, 'Cytoplasm_Volume', Name);
            end;
            if Cell_Volume > 0
                Name = cat(2, 'v(', num2str(Cell_Volume), ')');
                Flux = strrep(Flux, 'Cell_Volume', Name);
            end;
            if SY_Volume > 0
                Name = cat(2, 'v(', num2str(SY_Volume), ')');
                Flux = strrep(Flux, 'Synovium', Name);
            end;
                         
        case 2
            if Reactant_1 > 0 & ~isnan(Reactant_1)
                Name = cat(2, '(', S(Reactant_1).Name, ')');
                Flux = strrep(Flux, 'Reactant_1', Name);
                Name = cat(2, 'volume(', C(R1_Volume).Name, ')');
                Flux = strrep(Flux, 'R1_Volume', Name);
            end;
            if Reactant_2 > 0 & ~isnan(Reactant_2)
                Name = cat(2, '(', S(Reactant_2).Name, ')');
                Flux = strrep(Flux, 'Reactant_2', Name);
                Name = cat(2, 'volume(', C(R2_Volume).Name, ')');
                Flux = strrep(Flux, 'R2_Volume', Name);
            end;
            if Product_1 > 0 & ~ isnan(Product_1)
                Name = cat(2, '(', S(Product_1).Name, ')');
                Flux = strrep(Flux, 'Product_1', Name);
                Name = cat(2, 'volume(', C(P1_Volume).Name, ')');
                Flux = strrep(Flux, 'P1_Volume', Name);
            end;
            if Product_2 > 0 & ~isnan(Product_2)
                Name = cat(2, '(', S(Product_2).Name, ')');
                Flux = strrep(Flux, 'Product_2', Name);
                Name = cat(2, 'volume(', C(P2_Volume).Name, ')');
                Flux = strrep(Flux, 'P2_Volume', Name);
            end;
                        
            if Modifier_1 > 0 & ~isnan(Modifier_1)
                Name = cat(2, '(', S(Modifier_1).Name, ')');
                Flux = strrep(Flux, 'Modifier_1', Name);
                Name = cat(2, 'volume(', C(M1_Volume).Name, ')');
                Flux = strrep(Flux, 'M1_Volume', Name);
            end;    
            if Modifier_2 > 0 & ~isnan(Modifier_2)
                Name = cat(2, '(', S(Modifier_2).Name, ')');
                Flux = strrep(Flux, 'Modifier_2', Name);
                Name = cat(2, 'volume(', C(M2_Volume).Name, ')');
                Flux = strrep(Flux, 'M2_Volume', Name);
            end;
            if Modifier_3 > 0 & ~isnan(Modifier_3)
                Name = cat(2, '(', S(Modifier_3).Name, ')');
                Flux = strrep(Flux, 'Modifier_3', Name);
                Name = cat(2, 'volume(', C(M3_Volume).Name, ')');
                Flux = strrep(Flux, 'M3_Volume', Name);
            end;
        
            if Parameter_1 > 0 & ~isnan(Parameter_1)
                Name = cat(2, '(', P(Parameter_1).Name, ')');
                Flux = strrep(Flux, 'Parameter_1', Name);    
            end;
            if Parameter_2 > 0 & ~isnan(Parameter_2)
                Name = cat(2, '(', P(Parameter_2).Name, ')');
                Flux = strrep(Flux, 'Parameter_2', Name);   
            end;
            if Parameter_3 > 0 & ~isnan(Parameter_3)
                Name = cat(2, '(', P(Parameter_3).Name, ')');
                Flux = strrep(Flux, 'Parameter_3', Name);   
            end;
            if Parameter_4 > 0 & ~isnan(Parameter_4)
                Name = cat(2, '(', P(Parameter_4).Name, ')');
                Flux = strrep(Flux, 'Parameter_4', Name);   
            end;
            if Parameter_5 > 0 & ~isnan(Parameter_5)
                Name = cat(2, '(', P(Parameter_5).Name, ')');
                Flux = strrep(Flux, 'Parameter_5', Name); 
            end;
                        
            if PM_Volume > 0
                Name = cat(2, 'volume(', C(PM_Volume).Name, ')');
                Flux = strrep(Flux, 'PM_Volume', Name);
            end;
            if EM_Volume > 0
                Name = cat(2, 'volume(', C(EM_Volume).Name, ')');
                Flux = strrep(Flux, 'EM_Volume', Name);
            end;
            if EL_Volume > 0
                Name = cat(2, 'volume(', C(EL_Volume).Name, ')');
                Flux = strrep(Flux, 'EL_Volume', Name);
            end;
            if CY_Volume > 0
                Name = cat(2, 'volume(', C(CY_Volume).Name, ')');
                Flux = strrep(Flux, 'Cytoplasm_Volume', Name);
            end;
            if Cell_Volume > 0
                Name = cat(2, 'volume(', C(Cell_Volume).Name, ')');
                Flux = strrep(Flux, 'Cell_Volume', Name);
            end;
            if SY_Volume > 0
                Name = cat(2, 'volume(', C(SY_Volume).Name, ')');
                Flux = strrep(Flux, 'Synovium', Name);
            end;
        
    end;    % switch format
    clear Name;
    
    
    % Write flux expression to appropriate equation buffer (ODE(#).eqn):
    if Reactant_1 ~= 0
        switch format
            case 1
                if isempty(S(Reactant_1).ConstantSpecies) | S(Reactant_1).ConstantSpecies == 0
                    ODE(Reactant_1).usage = ODE(Reactant_1).usage + 1;
                    ODE(Reactant_1).eqn = cat(2, ODE(Reactant_1).eqn, sprintf(' - (%s)/v(%g)', Flux, R1_Volume));
                end;
            case 2
                if isempty(S(Reactant_1).ConstantSpecies) | S(Reactant_1).ConstantSpecies == 0
                    ODE(Reactant_1).usage = ODE(Reactant_1).usage + 1;
                    ODE(Reactant_1).eqn = cat(2, ODE(Reactant_1).eqn, sprintf(' - (%s)/volume(%s)', Flux, S(Reactant_1).Location));
                end;
        end;
    end;
    if Reactant_2 ~= 0
        switch format
            case 1
                if isempty(S(Reactant_2).ConstantSpecies) | S(Reactant_2).ConstantSpecies == 0
                    ODE(Reactant_2).usage = ODE(Reactant_2).usage + 1;
                    ODE(Reactant_2).eqn = cat(2, ODE(Reactant_2).eqn, sprintf(' - (%s)/v(%g)', Flux, R2_Volume));
                end;
            case 2
                if isempty(S(Reactant_2).ConstantSpecies) | S(Reactant_2).ConstantSpecies == 0
                    ODE(Reactant_2).usage = ODE(Reactant_2).usage + 1;
                    ODE(Reactant_2).eqn = cat(2, ODE(Reactant_2).eqn, sprintf(' - (%s)/volume(%s)', Flux, S(Reactant_2).Location));
                end;
        end;
    end;
    if Product_1 ~= 0
        switch format
            case 1
                if isempty(S(Product_1).ConstantSpecies) | S(Product_1).ConstantSpecies == 0
                    ODE(Product_1).usage = ODE(Product_1).usage + 1;
                    ODE(Product_1).eqn = cat(2, ODE(Product_1).eqn, sprintf(' + (%s)/v(%g)', Flux, P1_Volume));
                end;
            case 2
                if isempty(S(Product_1).ConstantSpecies) | S(Product_1).ConstantSpecies == 0
                    ODE(Product_1).usage = ODE(Product_1).usage + 1;
                    ODE(Product_1).eqn = cat(2, ODE(Product_1).eqn, sprintf(' + (%s)/volume(%s)', Flux, S(Product_1).Location));
                end;
        end;
    end;
    if Product_2 ~= 0
        switch format
            case 1
                if isempty(S(Product_2).ConstantSpecies) | S(Product_2).ConstantSpecies == 0
                    ODE(Product_2).usage = ODE(Product_2).usage + 1;
                    ODE(Product_2).eqn = cat(2, ODE(Product_2).eqn, sprintf(' + (%s)/v(%g)', Flux, P2_Volume));
                end;
            case 2
                if isempty(S(Product_2).ConstantSpecies) | S(Product_2).ConstantSpecies == 0
                    ODE(Product_2).usage = ODE(Product_2).usage + 1;
                    ODE(Product_2).eqn = cat(2, ODE(Product_2).eqn, sprintf(' + (%s)/volume(%s)', Flux, S(Product_2).Location));
                end;
        end;
    end;
    
end; % loop through reactions
%//////////////////////////////////////////////////////////////////////////




%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
% Use ODE(#).eqn to write ODEfile:


switch format
    case 1
        fprintf(fid, 'function dydt = ODEfile(t, y, flag, p, v)\n\n');
    case 2
        fprintf(fid, 'function dydt = ODEfile_text(t, y, flag, p, v)\n\n');       
end;

fprintf(fid, 'dydt = zeros(%g,1); %% pre-allocate memory\n\n', size(S,2));


% This part writes the equations for all the Species that don't have
% SubSpecies or 'dydt' on the right hand side of the equation 
SpeciesThatNeedToBeListedLast = [];
for i=1:size(S,2)
    
    if (ODE(i).usage == 0) | S(i).ConstantSpecies == 1
        ODE(i).eqn = cat(2, ODE(i).eqn, sprintf('0'));
    end;
    
    if isempty(S(i).SubSpecies) & isempty(strfind(ODE(i).eqn, 'dydt'))
            fprintf(fid, '%% d(%s)/dt:\ndydt(%g,1) = %s;\n%% # of rxns: %g\n\n\n', S(i).Name, i, ODE(i).eqn, ODE(i).usage);
    else
        SpeciesThatNeedToBeListedLast = cat(2,SpeciesThatNeedToBeListedLast, i);
    end;
end;



% This writes the equations for the CompoundSpecies and those with 'dydt in
% the right hand side of equation.
fprintf(fid, '%% Compound Species and species with ''dydt'' on the right hand side of equation ...\n\n');

for i=SpeciesThatNeedToBeListedLast
    fprintf(fid, '%% d(%s)/dt:\ndydt(%g,1) = %s;\n%% # of rxns: %g\n\n\n', S(i).Name, i, ODE(i).eqn, ODE(i).usage);
end;



fprintf(fid,'return;\n');
status = fclose(fid);
fprintf('\n%s COMPLETED! \n', filename);
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

return;
