#
# Model visualization, fitting, and assessing
#

# clear out the workspace
rm(list=ls())

# go to where the code is
setwd('~/Dropbox/MultisensoryIntegration/Papers/McGurk_MTurk_Compare/IMRF_Educational_Talk/part3_modeling/')

# define a range of asynchronies to test (should correspond to your behavioral data)
asynchronies = c(-300, -267, -200, -133, -100, -67, 0, 67, 100, 133, 200, 267, 300, 400, 500)

# load the functions needed
source('sj_modeling.R')

# Let's explore the basic model
# Open sj_modeling.R and locate the function predict.asymmetric_threshold_model


yhat_st = predict.symmetric_threshold_model(c(200, 100), asynchronies)
plot.sj(yhat_st)

# let's plot a few curves to see how the parameters impact the predictions
# the function p.stm is used here to make the code shorter, but it's the same as predict.asymmetric_threshold_model
plot.sj(p.stm(c(100, 100)), main='Single Threshold Model')
plot.sj(p.stm(c(200, 100)), add=TRUE, col='dodgerblue3')
plot.sj(p.stm(c(400, 100)), add=TRUE, col='orangered')
# now fix the threshold and change the sensory noise

# now that we've visualized the model, let's try and fit it to some actual data!!

# first we need some data
async_data = read.csv('sync_data.csv', row.names=1)
async_data = as.matrix(async_data)

# let's just fit a single subject
y = async_data[7,]
plot.sj(y, type='p')


# Check out cost.sse and optim.model_sse in simple_sj.R
best_fit.st = optim.model_sse(y, asynchronies, c(100, 100),
                              predict.symmetric_threshold_model)

# what is returned?
best_fit.st

# NB: if you are getting fitted values that are very large/small or near the boundary
# it may be that you had a particularly bad guess and need to try more initial conditions,
# or switch optimization strategies, or the model is really really misspecified 

# NB: convergence = 0 is a good thing, but doesn't guarantee optimalit
# run the following line for more details
?optim

# now that we have our fitted parameters, get the predictions
fitted.st = predict.symmetric_threshold_model(best_fit.st$par, asynchronies)
plot.sj(fitted.st, col='dodgerblue3', add=TRUE)

# quantify the fit
cost.sse(y, fitted.st) # same as best_fit$value
sqrt(mean((y-fitted.st)^2))
cor(y, fitted.st)^2
mean(abs(y-fitted.st))

# for fun, pick another subject and checkout the fit (go back to Line 38 and change the row index)

# is this fit good?
# this is actually a hard question to answer. so let's not try yet
# Instead, let's build a more complex model and see if it beats the simpler model

# checkout predict.asymmetric_threshold_model in sj_modeling.R

yhat_at = predict.asymmetric_threshold_model(c(-200, 300, 100), asynchronies)
plot.sj(yhat_at)

# re-run lines 74-75 while changing parameters. 

# fit the data in the same
best_fit.at = optim.model_sse(y, asynchronies, c(-200, 300, 100),
                           predict.asymmetric_threshold_model)

fitted.at = predict.asymmetric_threshold_model(best_fit.at$par, asynchronies)

plot.sj(y, type='p') # actual data
plot.sj(fitted.st, col='dodgerblue3', add=TRUE)
plot.sj(fitted.at, col='orange', add=TRUE)

#draw the fitted syncrhony window
abline(v=best_fit.st$par[1] * c(-1, 1), col='dodgerblue3', lwd=2, lty=2)
abline(v=best_fit.at$par[1:2], col='orange', lwd=2, lty=2)

cost.sse(y, fitted.st)
cost.sse(y, fitted.at)

# could keep trying with different subjects....

# Let's make one more model, that will take into account subjects having a bias toward
# a particular alternative
#   
#   predict.asymmetric_threshold_bias_model
#

yhat_atb = predict.asymmetric_threshold_bias_model(c(-100, 250, 120, .2), asynchronies)
yhat_at = predict.asymmetric_threshold_model(c(-100, 250, 120), asynchronies)
plot.sj(yhat_atb, main='Double Threshold w/ Bias', col='darkgreen')
plot.sj(yhat_at, add=TRUE, col='orange')


# fit the models!
best_fit.atb = optim.model_sse(y, asynchronies, c(-200, 250, 100, .2), predict.asymmetric_threshold_bias_model)
fitted.atb = predict.asymmetric_threshold_bias_model(best_fit.atb$par, asynchronies)

#recall the data
plot.sj(y, type='p')

plot.sj(fitted.st, col='dodgerblue3', add=TRUE)
plot.sj(fitted.at, col='orange', add=TRUE)
plot.sj(fitted.atb, col='darkgreen', add=TRUE)


#
# There are other models that incorporate a guessing term, feel free to chek them out, 
# they don't do well on the current data
#

# how would you fit the models across subjects?
fits.st = apply(async_data, 1, optim.model_sse,
                 asynchronies,
                 c(300, 100),
                 predict.symmetric_threshold_model)
pars.st = sapply(fits.st, get_pars)
yhat.st = apply(pars.st, 2, predict.symmetric_threshold_model, asynchronies)

# double-threshold, unbiased model
fits.at = apply(async_data, 1, optim.model_sse,
                asynchronies,
                c(-200, 250, 100),
                predict.asymmetric_threshold_model)

pars.at = sapply(fits.at, get_pars)
yhat.at = apply(pars.at, 2, predict.asymmetric_threshold_model, asynchronies)

# double-threshold, with bias model
fits.atb = apply(async_data, 1, optim.model_sse,
                asynchronies,
                c(-200, 250, 100, .2),
                predict.asymmetric_threshold_bias_model)

pars.atb = sapply(fits.atb, get_pars)
yhat.atb = apply(pars.atb, 2, predict.asymmetric_threshold_bias_model, asynchronies)


par(mfrow=c(4,4), mar=rep(1,4))
for(ii in 1:16) {
    plot.sj(async_data[ii,], type='p', axes=F)
    plot.sj(yhat.st[,ii], col='dodgerblue3', add=TRUE)
    plot.sj(yhat.at[,ii], col='orange', add=TRUE)
    plot.sj(yhat.atb[,ii], col='darkgreen', add=TRUE)
    box()
}

# now to say that we don't need the bias term, we need a cost function that takes into
# account the number of parameters in each model
# BIC needs the data as counts
n_trials = 12
async_counts = async_data * n_trials
bic_matrix = array(NA, c(16, 3))
for(ii in 1:16) {
    bic_matrix[ii,1] = cost.BIC(async_counts[ii,], yhat.st[,ii], rep(n_trials, length(asynchronies)), n.par=2)
    bic_matrix[ii,2] = cost.BIC(async_counts[ii,], yhat.at[,ii], rep(n_trials, length(asynchronies)), n.par=3)
    bic_matrix[ii,3] = cost.BIC(async_counts[ii,], yhat.atb[,ii], rep(n_trials, length(asynchronies)), n.par=4)
}

# reset the plotting window
dev.off()

# first let's compare ST to DT, taking 1 minus 2 so positive diff means HIGHER COST for ST,
# and thus a better fit for DT, so color those orange
bic_diff <- bic_matrix[,1] - bic_matrix[,2]
barplot(bic_diff,
        col=ifelse(bic_diff < 0, 'dodgerblue3', 'orange'),
        ylab='BIC Difference', las=1, border=NA)
abline(h=0, col='lightgray')

# pretty clear that the symmetric thresholds are not good, 
# although 2 subjects are equivocal

# notice that we never calculated p-values to say certain conditions are > than others
# compare models that make different assumptions, reject the one that doesn't fit


# now compare DT to DT w/ bias. do 3 minus 2, so positive means 
bic_diff <- bic_matrix[,3] - bic_matrix[,2]
barplot(bic_diff,
        col=ifelse(bic_diff < 0, 'darkgreen', 'orange'),
        ylab='BIC Difference', las=1, border=NA)
abline(h=0, col='lightgray')

# ways to summarize the model comparison
sum(bic_diff)
mean(bic_diff)
wilcox.test(bic_diff)

# only thing you really would need to do is 
# make sure the results aren't sensitive to initial conditions


#
# topics to explore for the future
#
# cross validation / generalization testing
#       leave out some asynchronies, see how well the models predict 
#
# fake-data testing / model recovery testing
#       generate data according to a model (taking into account # of trials) and then
#       fit models
#

