# how likely are you to find p < 0.05 if the effect is XX%,
# using bootstrapped McGurk values and some simplifying assumptions about effects
source('basic_plotters.R')

yax <- function() draw.axis(2, 0:2/2, labels=0:2/2 *100)

plot.grp_bplot <- function(df_, cols = c('orange', 'dodgerblue3')){
    y <- df_ %>% do_aggregate(pF_hat ~ group, m_se)

    x <- barplot(ylim=0:1, y$pF_hat[,'mean'], border=NA, col=getAlphaRGB(cols, 150), las=1, axes=F)

    ebars.y(x, y$pF_hat[,'mean'], y$pF_hat[,'se'], col=cols, code=0, lwd=1)

    yax()

    return(x)
}

# PDF version of the above plot
pdf.pgb <- to_pdf(plot.grp_bplot, w=1.25, h = 1.75, mar=rep(0.5, 4))

plot.power_hist <- function(result, pop_delta, ymax=2000, breaks=seq(-1, 1, by=0.005), col.ns = 'gray50', col.sig='dodgerblue3', xlim=c(-.3, .55)) {
    ps <- result[,'pval']
    effs <- result[,2]

    draw.hist <- function(x, col, lwd = 4, add=FALSE, min_count = round(.001 * ymax)) {
        res <- hist(x, plot=FALSE, breaks=breaks)
        if(!add) plot.clean(xlim, c(0,ymax))

        rc <- smooth(res$counts, twiceit = FALSE)
        rc[rc <= 5] = NA

        keep.ind <- res$mids >= -.3 & res$mids <= .55

        points(res$mids[keep.ind], rc[keep.ind], type='h', lwd=1, lend=2, col=col)

        return(max(res$counts))
    }

    draw.hist(effs, getAlphaRGB(col.ns, alpha=150))
    est_eff <- effs[ps<0.05]
    draw.hist(est_eff, col=getAlphaRGB(col.sig, 150), add=TRUE)

    abline(v=pop_delta, lwd=1.5, lty=2, col='gray30')

    draw.axis(1, at=-1:2 / 4)
    draw.axis(2, at=c(0, ymax), tcl=0)
}

#pdf version of the above plot
ph.pdf <- to_pdf(plot.power_hist, 2.15, 1.75, mar=rep(0.5, 4))


setup_graph <- function(..., plot_var = 'power', do_labels=TRUE) {
    if(plot_var=='power') {
        plot.clean(20:300, 0:1, ...)
        abline(h=c(0.05, .8), col='gray50', lty=c(2,1))

        lbl <- 100 * (at <- 0:5/5)
        if(!do_labels) lbl <- NA

        draw.axis(2, at, labels=lbl)
    } else {
        plot.clean(20:300, 1:8, log='y', ...)
        draw.axis(2, at=c(1, 2, 4, 8))
        abline(h=1, lty=2, col='gray50')
    }

    # X axis is sample size
    lbl <- (at <- c(20, 100, 200, 300))
    if(!do_labels) lbl <- NA

    draw.axis(1, at, labels=lbl)
}

lines.power <- function(pcurve, col, var='power', ...) {
    lines(pcurve$N, pcurve[[var]], type='o', col=col, lwd=2, pch=16, cex=0.6, ...)
}

line.drawer <- function(colors) {
    ii <- 1
    return (function(pcurve, var='power') {
        lines.power(pcurve, colors[ii], var=var)

        ii <<- ii + 1

        if(ii > length(colors)) ii <<- 1

        invisible(ii)
    })
}



# write everything to PDF
# See below for individual plotting
write_out_graph <- function(pcurves, cols, name) {
    as_pdf(name, w=2.75, h=2.75, {
        par(mar=rep(0.5,4))
        setup_graph(do_labels=FALSE)
        mapply(lines.power, pcurves, cols)
    })
}


combine_pc <- function(p1, p2) {
    p3 <- p1
    n1 <- unique(p1$n.rep)
    n2 <- unique(p2$n.rep)

    stopifnot(length(n1) == 1, length(n2) ==1, nrow(p1) == nrow(p2))

    nT <- n1 + n2

    p3$n.rep <- nT

    p3[,-(1:7)] <- (p3[,-(1:7)] * n1 + p2[,-(1:7)]*n2) / nT

    # any NAs in p1, copy them over from p2
    p3[is.na(p3)] <- p2[is.na(p3)]

    return(p3)
}

make.summary_plot <- function(fname, pmat, cols, split_var, write_out=TRUE, plot_var='power') {
        plotter <- line.drawer(cols)
        if(write_out) {
            as_pdf(fname, w=2.75, h=2.75, {
                par(mar=rep(0.5, 4))
                setup_graph(plot_var=plot_var)
                pmat %>% split((.)[[split_var]]) %>% lapply(plotter, var=plot_var)
            })
        }

        # always plot
        setup_graph(plot_var=plot_var, main=fname)
        pmat %>% split((.)[[split_var]]) %>% lapply(plotter, var=plot_var)
    }

if (FALSE) {

    power_curves <- list.files(pattern='csv') %>% sort %>% sapply(read.csv, USE.NAMES=TRUE, simplify=FALSE)

    names(power_curves)

    # combine the separate simulations into a single file
    inc_delta <- combine_pc(power_curves[[1]], power_curves[[2]])
    inc_stim <- combine_pc(power_curves[[3]], power_curves[[4]])
    inc_trial <- combine_pc(power_curves[[5]], power_curves[[6]])

    # add n_stim==1 to pc2 and n_trials==10 to pc3
    inc_stim <- rbind(inc_delta %>% subset((.)$pop_eff == 0.1), inc_stim)
    inc_trial <- rbind(inc_delta %>% subset((.)$pop_eff == 0.1), inc_trial)

    ## colors palettes
    delta_cols <- RColorBrewer::brewer.pal(7, 'Blues')[4:7]
    stim_cols <- RColorBrewer::brewer.pal(7, 'Reds')[2:6]
    trial_cols <- RColorBrewer::brewer.pal(5, 'Greens')[3:5]

    ### Power plots
    make.summary_plot('increasing_delta_power', inc_delta,
        delta_cols, 'pop_eff')

    make.summary_plot('increasing_stimuli_power', inc_stim,
        stim_cols, 'n_stim')

    make.summary_plot('increasing_trials_power', inc_trial,
        trial_cols, 'n_trials')

    ### Inflation factor graphs
    make.summary_plot('increasing_delta_eff_inf', inc_delta,
        delta_cols, 'pop_eff', plot_var='mean_abs_ratio')

    make.summary_plot('increasing_stimuli_eff_inf', inc_stim,
        stim_cols, 'n_stim', plot_var='mean_abs_ratio')

    make.summary_plot('increasing_trials_eff_inf', inc_trial,
        trial_cols, 'n_trials', plot_var='mean_abs_ratio')

}

if(FALSE){

    # make bar graph of a-only results
    sylls <- c('ba', 'da', 'ga')
    aonly <- read.csv('../../data/auditory_only.csv')
    sbj.corr <- aonly %>% split((.)$sid) %>% sapply(function(sbj) {
        df <- sbj %>% do_aggregate(cbind(ba,da,ga) ~ syllable, mean)
        diag(as.matrix(df[,-1])) %>% sum / sum(df[,-1])
    })

    is_auditory <- function(stim) stim %>% str_split('_') %>% unlist %>% extract2(4) %>% equals('blank')

    load('~/Dropbox/MultisensoryIntegration/Experiments/large_scale_mcgurk/mTurk_results/unisensory.RData')
    uni_mats$mats %>% sapply(function(mat) {
        df <- mat[rownames(mat) %>% sapply(is_auditory),] %>% data.frame
        df$syllable <- rownames(df) %>% sapply(function(stim) stim %>% str_split('_') %>% unlist %>% extract2(3) %>% as.character)

        df[,1:3] <- df[,1:3]/2

        mean(c( df$Ba[df$syllable=='C1'],
                df$Da[df$syllable=='C2'],
                df$Ga[df$syllable=='C3']))
    }) -> mturk_aonly_acc

    joint_df <- data.frame('acc' = c(sbj.corr, mturk_aonly_acc),
        'grp' = rep(c('IRL', 'MTRK'), times=c(length(sbj.corr), length(mturk_aonly_acc))))

    t.test(acc ~ grp, data=joint_df, var.equal=TRUE)
    y <- joint_df %>% do_aggregate(acc ~ grp, m_se)
    cols <- c('orange','brown')

    as_pdf('../../figures/sub_figures/compare_aonly.pdf', w=1.15, h=2, {
        par(mar=rep(0.75,4))
        x <- barplot(ylim=0:1, y$acc[,'mean'], border=NA, col=getAlphaRGB(cols, 150), las=1, axes=F)
        ebars.y(x, y$acc[,'mean'], y$acc[,'se'], col=cols, code=0, lwd=1)
        draw.axis(2,at=0:2/2, labels=NA)
    })


}

if(FALSE) {

    map_to_col <- function(dl) {
        delta_cols[as.integer(round(dl*10))]
    }

    ###
    # quick compare power to eff inflat
    plot.clean(1:8, c(0,1.1), log='x')
    inc_delta %>% split((.)$pop_eff) %>% lapply(function(pw) {
        pef <- unique(pw$pop_eff)
        if(pef<.4) {
            abs <- (pw[,'mean_abs_ratio'])
            pow <- (pw[,'power'])

            points(abs, pow, col=getAlphaRGB(map_to_col(pef), 200), cex=log(pw$N)*.75, pch=16, type='p')
            # lines(abs, pow, col=getAlphaRGB(map_to_col(pef), 255), lwd=2, type='l')
        }
    })
    draw.axis(1, at=c(1,2,4,8))
    draw.axis(2, at=0:5/5, labels=0:5/5 * 100)
}
