# python implementation of gibson-bruck optimization of
# gillespie method.
#
# lawrence david, 2006. ldavid@mit.edu
# sonia timberlake, 2006. soniat@mit.edu

from gillespie_helper import *
import gillespie_helper
import math
import random
import sys

# reaction parameters will be stored using dictionaries of
# dictionaries.  reactions parts are stored in three places: products
# live in product_dict, reactants live in substrate_dict, and
# propensities live in c_dict.  all three of these dictionaries are
# indexed by the same reaction names.
#
# examples of the dictionaries:
# consider RXN1: A + 2B -> C with propensity 0.01
#
# substrate_dict['RXN1'] = {'A':1, 'B':2}
# product_dict['RXN1'] = {'C':1}
# c_dict['RXN'] = 0.01
#
# counts_dict stores species' abundances.

substrate_dict, product_dict, c_dict = ReadInputFile(sys.argv[1])
counts_dict = ReadCountsFile(sys.argv[2])

# creates a "reaction graph" from the system input file
# update_dict[rxn_fired] contains keys to every rxn it will affect.
update_dict = {}  # key: rxn that fired    value: rxns needing updating
for rxn_fired in substrate_dict:
    update_dict[rxn_fired] = {}   # will be list of rxn's to update
    listokeys = substrate_dict[rxn_fired].keys() + product_dict[rxn_fired].keys() 
    for sp in listokeys:
        if substrate_dict[rxn_fired].has_key(sp) and product_dict[rxn_fired].has_key(sp):
            if product_dict[rxn_fired][sp] == substrate_dict[rxn_fired][sp]:
                continue  # no. of this reactant unchanged
        for other_rxn in substrate_dict:
             if substrate_dict[other_rxn].has_key(sp)==False:
                 continue
             update_dict[rxn_fired][other_rxn] = 1  # binary indicator

# lists storing time-steps and species concentrations at each step
t_list = [0]
s_list = [counts_dict.copy()]

# a very small number
eps = math.exp(-200)

# calculate both a_i and a
def GetA_i(substrate_dict,c_dict,counts_dict,update_after_mu):

    a_i = {}
    a = 0

    # here, we first calculate h and then take product with c
    for rxn in update_after_mu:
        a_i_rxn = 1
        substrate_rxn = substrate_dict[rxn]

        # calculate h, the reaction propensity by successively taking
        # product of all species' counts involved in this reaction.
        # note that in most cases, substrate_rxn[species] is 1 and
        # NChooseK(n,1) just equals n.  
        for species in substrate_rxn:
            n = counts_dict[species]
            k = substrate_rxn[species]
            if k > n:
                a_i_rxn = 0
                break
            if k == 1:
                a_i_rxn *= n
            else:
                a_i_rxn *= NChooseK(n,k)

        a_i_rxn *= c_dict[rxn]

        # store reaction probabilities in a dictionary
        a_i[rxn] = a_i_rxn    # update a_i

    return a_i

# calculate the initial a_i - have to "update" all rxn's a_i's
a_i_old = GetA_i(substrate_dict,c_dict,counts_dict,substrate_dict)

# some additional housekeeping to enable the gibson speedup
tau_i = {}
note11 = {}

mintau = 1000
# calculate the initial tau_i
for rxn in a_i_old.keys():
    ai = a_i_old[rxn]
    note11[rxn] = {}
    if ai == 0:
        note11[rxn]['t1'] = 0
        note11[rxn]['a'] = 0
        tau_i[rxn] = 1e50
        continue
    ri = random.random()
    taui = (1.0/(ai+eps)) * math.log(1.0/ri)
    tau_i[rxn] = taui 
    if taui < mintau:
        mu = rxn     # the putative first rxn 
        mintau = taui # reset the min
t_cur = mintau

t_max = 1000

# step through time, identifying the next reaction,
# updating species and propensities only as necessary
while t_list[-1] < t_max:

    # if there are no species left, the simulation should end
    if sum(a_i_old.values()) == 0:
        t_list.append(sys.maxint)
        break

    # sort the tau dictionary to find the next reaction
    if t_list[-1] != 0:
        t_cur = sys.maxint
        for i in tau_i:
            if tau_i[i] < t_cur:
                t_cur = tau_i[i]
                mu = i
        #inside_out = dict(map(lambda i: (i[1],i[0]), tau_i.items()))
        #t_cur = min(inside_out.keys())  # the rxn with the minimum tau
        #mu = inside_out[t_cur]

    # update the substrate and product species counts involved
    for species in substrate_dict[mu]:
        counts_dict[species] -= substrate_dict[mu][species]
    for species in product_dict[mu]:
        counts_dict[species] += product_dict[mu][species]

    # record the events
    t_list.append(t_cur)
    s_list.append(counts_dict.copy())

    # update a_i selectively
    a_i_new = GetA_i(substrate_dict,c_dict,counts_dict,update_dict[mu])

    # use updated propensities to update times as necessary
    # reuse the randomly generated waiting times for rxns that didn't fire 
    # but scale their scale & shift those times appropriately
    for rxn in update_dict[mu]:

        # first, some housekeeping
        # if a reaction has just become impossible, store its old values
        if a_i_new[rxn] == 0:
            if rxn == mu:
                note11[rxn]['a'] = 0
            else: 
                note11[rxn]['t1'] = t_cur
                note11[rxn]['a'] = a_i_old[rxn] # last non-zero propensity
                note11[rxn]['tau'] = tau_i[rxn]
                tau_i[rxn] = 1e200
                a_i_old[rxn] = 0
                continue

        if rxn == mu:
            a_i_old[mu] = a_i_new[mu]
            continue

        # if this rxn has just become possible
        elif a_i_old[rxn] == 0:
            lastNonzero_a = note11[rxn]['a']
            if lastNonzero_a == 0: # this is the first time its been possible
                r = random.random()
                tau_i[rxn] = taui = (1.0/(a_i_new[rxn]+eps)) * math.log(1.0/r) + t_cur
            else:
                t1 = note11[rxn]['t1']
                tau = note11[rxn]['tau']
                tau_i[rxn] = (lastNonzero_a/(a_i_new[rxn]+eps)) * (tau - t1) + t_cur
            continue

        # otherwise, if we're not dealing with special cases:
        if rxn != mu:
            # insert your code here for updating the reaction time
            # don't forget to update the propensity also!

    # end of for loop

    # choose a new random number for mu, the rxn that fired
    r = random.random()   
    tau_i[mu] = (1.0/(a_i_old[mu]+eps)) * math.log(1.0/r) + t_cur

# end of while loop

# write the results to an output file
names = s_list[0].keys()
output_file = open(sys.argv[3],"w")
output_file.write("time ")
for k in s_list[0].keys():
    output_file.write("\t" + k)
output_file.write("\n")

# specify how many data points to use
data_points = int(sys.argv[4])
interval = len(t_list)/data_points

if interval < 1:
    interval = 1

for i in range(0,len(t_list),interval):
    output_file.write(str(t_list[i]))
    for j in names:
        output_file.write("\t" + str(s_list[i][j]))
    output_file.write("\n")
output_file.close()

