library('parallel')

#go to the right directory
setwd('~/Desktop/cims_code_pack/')

# Load the fitting functions
source('cims.R')

# rows are subjects, columns are asynchronies, cells are counts of Sync = Yes
count_mat = as.matrix(read.csv(file='data.csv') )

# number of trials at each asynchrony
max_count = 12

# this must match the column order in count_mat
asyncs = c(-300, -267, -200, -133, -100, -67, 0, 67, 100, 133, 200, 267, 300, 400, 500)

#
# Note that the number of asynchronies must be an integer multiple of the number of columns in the dataset.
# When there are more columns than asynchronies, the code assumes you have multiple conditions. 
# For example, data.csv has 60 columns, so the code assumes 60 / 15 = 4 conditions
#
# The CIMS model fits a separate sigma (sensory noise) term for each condition
# The Gaussian model fits a separate mean and sigma for each condition
#

##
#	MODEL FITTING
##

# Create a cluster to speed up computation
cl = makeCluster(detectCores())

# obtain CIMS fitted values
# each repetition takes about 15s on a circa 2012 macbook pro
cims.model = cims(n.reps=512)

# obtain Gaussian fitted values
gauss.model = gauss(n.reps=512)

##
#	MODEL COMPARISON
##

# test log likelihoods for each condition
within_condition_tests(cims.model, gauss.model)

# test total log likelihood
across_condition_tests(cims.model, gauss.model)

##
#	MODEL PLOTTING
##

# grab predicted values
cims.p = predict(cims.model)
gauss.p = predict(gauss.model)

#create proportion data from count data
actual = count_mat / max_count

#create condition indicies
c1 = 1:15
c2 = 16:30

# plot actual + fitted data for conditions 1 and 2
plotMeanWithFitted(asyncs, actual[,c1], cims.p[,c1], gauss.p[,c1])
plotMeanWithFitted(asyncs, actual[,c2], cims.p[,c2], gauss.p[,c2])

# calculate negative log likelihood for each condition
cims.nlnL = -logLik(cims.model, separate=T)
gauss.nlnL = -logLik(gauss.model, separate=T)

# plot the negative log likelihoods for conditions 1 and 2
plotModelComparison(cims.nlnL[,1], gauss.nlnL[,1], ylim=c(50, 105), yat=c(50, 75, 100))
plotModelComparison(cims.nlnL[,2], gauss.nlnL[,2], ylim=c(50, 105), yat=c(50, 75, 100))
