#!/usr/bin/env python
# encoding: utf-8

# The MIT License
#
# Copyright (c) 2011 Wyss Institute at Harvard University
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# http://www.opensource.org/licenses/mit-license.php

"""
scaffold-permutation-analysis.py

Created by Shawn Douglas and Adam H. Marblestone on 2010-06-30.
Modified by Adam H. Marblestone and the Harvard BioMod Team on 2011-06-09.

Copyright (c) 2011. All rights reserved.

Input: -caDNAno-generated json files in the designDir, 
       and scaffold sequences in the sequenceDir
       
       -set of staples to match

Scaffold is chosen according to the design filename. 
E.g. "7560.monolith.json" will use the sequence found
in the p7560.txt file in the sequenceDir.
"""

import sys
import os
import string
import json
import pickle
import shutil
import time

designDir = "designs"
sequenceDir = "scaffolds"

staplesToMatch = {"Sphere-North-1":"AAAGAACTGGAATATCGCGTTTTAATTCGACTACTAATAG",
"Sphere-South-1":"ATTCCACACAACCGACAAAAGGTAAAGTAACGACTTGCGG",
"Sphere-North-2":"TAGTAGCATTAGCCTTAAATCAAGATTAGTGGAATACCCA",
"Sphere-South-2":"GAGGTTTTGAATTCTCCGTGGGAACAAACGTCCGCTCACA",
"Sphere-North-3":"TACATAACGAGGAAGCCCGAAAGACTTCACATGATTAAG",
"Sphere-South-3":"TATGTAAATGTAATAAGAGAATATAAAGTACATACGAGC",
"Sphere-North-4":"ACTCCTTATTTAGCCCCCTTATTAGCGTTCTAATGCAGA",
"Sphere-South-4":"CGGAAGCATAAGGGCGAAAAACCGTCTATTATATAACTA",
"Sphere-North-5":"ATCAAGAGTAATCTTGTGAGATTTAGGAATA",
"Sphere-South-5":"CGAATTATTCATTTCATTTTAACCTCCGGCT",
"Sphere-North-6":"CCACATTCAATGCCATCTTTTAGGCTGGCTGACCTTC",
"Sphere-South-6":"TAGGTTGGGTCAGGGCGATGGCAAAATCGCGCAGAGG",
"Sphere-North-7":"AGCAAACGTAGAGAAACCGAGGAAACGCAATAATAACTGCTATTTTGCAC",
"Sphere-South-7":"TGGGGTGCCTAAAGCTGTTTCCTGTGTGAAATTGTTAGCGGATTGACCGT",
"Sphere-North-8":"GAACCAGAGCCACGTTTTCATCGGCATTTTCGGTCAACGCAGTATGTT",
"Sphere-South-8":"CACCCAAATCAAAAAGAACGTGGACTCCAACGTCAAAAGTGTAAAGCC",
"Sphere-North-9":"AAACAGTTACCAGGCGCATCATAATCAAAATCACCG",
"Sphere-South-9":"ACAGGAACATACCAAGTTACCCACTACGTGAACCAT",
"Sphere-North-10":"TACATAAAGTCAGACTGTAGCGCCACCGACAGTGCCCGTAT",
"Sphere-South-10":"GCTAACTCAAGAGTCCACTATTGTTTTTAAAGGGATTTTAG",
"Sphere-North-11":"CCAGCTACAATTTGTTTTAGCGAACCTCCTTCTGTCCAGACGACGGCATTT",
"Sphere-South-11":"AATGGGATAGGTCAAAGGTGGCATCAATTGCTTCAAAGCGAACCATCAAAA",
"Sphere-North-12":"AGATGAACGGTGTACAGAATGCCCCCTGC",
"Sphere-South-12":"TCGCCTGATTGCTTTGAGGTACGCCAGAA",
"Sphere-North-13":"CTATTTCGGAACAGAGCCACCACCCTCATTTCTTTTTCATGAGGAA",
"Sphere-South-13":"TCCTGAGAAGTGACGCTCAATCGTCTGAAATAAAGTTTGAGTAACA",
"Sphere-North-15":"GTTTCCATTAAACGGGCAACTTTGAAAGAGGAC",
"Sphere-South-15":"TTATCATTTTGCGGAAGAGAAACAATAACGGAT",
"Sphere-North-16":"AAACAGCTTGGAGGCTTTGAGGACTAAAGATCAGGGATAG",
"Sphere-South-16":"TGGTCAGTTGCCCGAACGTTATTAATTTTAGGATTATTTA",
"Sphere-North-17":"CAAGCCCAATTTTTCTGTATGTGAATTTCTT",
"Sphere-South-17":"CATTGGCAGAGATAGCCCTACAATCAATATC",
"Sphere-North-18":"ATTGCGTGCTTTCGAGGGGATTTTGCTAAGGA",
"Sphere-South-18":"GTATTAATCAAACCCTAAACATCGCCAGGTCA",
"Sphere-North-19":"AAAGAAACGCATTAAGAAAAGTAAGCAGATAGCCGAACAACGCTAACGAG",
"Sphere-South-19":"GCGCTCACTGCGGATCCCCGGGTACCGAGCTCGAATTGGCGCATCGTAAC",
"Sphere-North-20":"AGCCGCCACCCTCACAGAATCAAGTTTGCCTTTAGCGGTGGCAACATATA",
"Sphere-South-20":"TAAAGCACTAAATTGAGTGTTGTTCCAGTTTGGAACACATTAATTGCGTT",
"Sphere-North-21":"GTCAGTGCCTTGAGTAGAACCGCCTCCCTCAG",
"Sphere-South-21":"AAACAGGAGGCCGATTTGGGGTCGAGGTGCCG",
"Sphere-North-22":"CACGGAATCGTAATCAGTAGCGAGAACCAATAAGTTTTAACGGG",
"Sphere-South-22":"CCAGTCGGGCCCGAGATAGGGTCGGAACAATCAGAGCGGGAGCT",
"Sphere-North-23":"CGTCTTTCCAGAAGATATAGAAGGCTTGCAGAACGCGCCTGTTCGCCAT",
"Sphere-South-23":"CGTGCATCTGCCTAACCTGTTTAGCTAAGGATTAGAGAGTACTTTACCC",
"Sphere-North-24":"CCATATTATTTATTCATCGTAGGAATCACAAGAAAAATAATATGTATAAA",
"Sphere-South-24":"TCAGGAAGATCGTAGATTTAGTTTGACCGTCATTTTTGCGGATTTCAGAA",
"Sphere-North-25":"ATATGGTGCCGGAAACGTCACCGCCACACATGGCTTTTGATGA",
"Sphere-South-25":"GGCCAACTCCGAAATCGGCAAACGGGGTGACGAGCACGTATAA",
"Sphere-North-26":"TACAGGAGTGTGAGGCTGAGACTCCTCAAGAGAAGGATTAGTTTAGTACC",
"Sphere-South-26":"CGTGCTTTCCTAAAGAGTCTGTCCATCACGCAAATTAACCCGCCAGCCAT",
"Sphere-North-27":"GCCACCCTCAGAACGTAACACTGAGTTTCGTCACCAGTACTTTGTCGT",
"Sphere-South-27":"TGCAACAGGAAAAAACCAGTAATAAAAGGGACATTCTGGCTGAATGGC",
"Sphere-North-28":"CTTTCCAGACTTTCAACAGTTTCAAGGAACAACTAAAACAACGTTAGT",
"Sphere-South-28":"TATTAGTCAATACCGAACGAACCAAGAGGTGAGGCTTAAATTTAATGC",
"Sphere-North-29":"AAATGAAAGGAACCCATGTACCCGCCACCCTCAGA",
"Sphere-South-29":"GCGAACTTTCACCAGTCACACGCGCTCATGGAAAT",
"Sphere-North-30":"ACCGCCACCCTCCTATTATTCTGAAA",
"Sphere-South-30":"ACCTACATTTTGTTTTTATAATCAGT",
"Sphere-North-31":"CATGAAAGTATTAAACTGGTGCCACCCTCAGAGCCA",
"Sphere-South-31":"GAGGCCACCGAGTACGTTAGCCTAAAGGGAGCCCCC",
"Sphere-North-32":"CCACCCTCAGAGCCAATGAAACCATCGATAGCAGCACAAGTTTATTTTG",
"Sphere-South-32":"GATTTAGAGCTTGAATCCCTTATAAATCAAAAGAATAGAAACCTGTCGT",
"Sphere-North-33":"TCACAATCAATAAAACAATGAAATAGCAATAGCTATCACAAAATAAACAG",
"Sphere-South-33":"GCCAGCTGCATTGGCCAGTGCCAAGCTTGCATGCCTGGACAGTATCGGCC",
"Sphere-North-34":"GTTTAACGTCAAAAAGTACCGCACTCATAACCAATCAATAATCAGCCTGT",
"Sphere-South-34":"CTGGTGCCGGAAATCATTCCATATAACAATGCTGTAGCTCAACCATAAAT",
"Sphere-North-35":"CAACCGATCCAGCAAAATCACCTGACAGGTCTCTGAATTTACCG",
"Sphere-South-35":"CCAGGGTGGTTTGCCCCAGCAGGAAGAAGCGCCGCTACAGGGCG",
"Sphere-North-36":"TTCCAGTAAGCGTCATCAGAACCACCACCAGAGC",
"Sphere-South-36":"CGTACTATGGTTGCTTAAAGCCGGCGAACGTGGC",
"Sphere-North-37":"CGCCGCCAGCATAGTAGCACCATTACCATTAGCAAGTTACCAGCGCCA",
"Sphere-South-37":"GAGAAAGGAAGGGCGAAAATCCTGTTTGATGGTGGTGCGCGGGGAGAG",
"Sphere-North-38":"AAGACAAAAGGGAACCCACAAGAATTGAGTTAAGCCCGAAACGATTTTTT",
"Sphere-South-38":"GCGGTTTGCGTATAACGCCAGGGTTTTCCCAGTCACGTCCGGCACCGCTT",
"Sphere-North-39":"GGCGCGAGCTGAACGTTGGTGTAGATGCGTAATCATGGTCATTGAGTGA",
"Sphere-South-39":"AGAACGCGAGGCTATCCTGAATCTTACCAAAGTTACCAGAAGAAATACA",
"Sphere-North-40":"AGATTAAGCCAAAAGGAATTACATCAGTACAAGAACCGGATATT",
"Sphere-South-40":"TCGAGCCAGCTGATGCAAATCCCTACCTATTACCTGAGCAAAAG",
"Sphere-North-41":"CATTACCCAAAACGGTCAATCATAAGGGAACCGAACTGACTAAAATACGT",
"Sphere-South-41":"AAGATGATGAAATATACAGTAACAGTACCTTTTACATCGGCAAAGAAACC",
"Sphere-North-42":"AATGCCACTACGAAGTCGGAACGAGGGTAGCAACGGCTACAATACCGAT",
"Sphere-South-42":"ACCAGAAGGAGCGGATTCGACAACTCGTATTAAATCCTTTGGCAAATCA",
"Sphere-North-43":"AGTTGCGCTATCGGTTTATCAGCTAATAATAATTAATTGCGACAATG",
"Sphere-South-43":"ACAGTTGATTGCTGAACCTCAAATACACCGCCTGTCACCAAGGAATT",
"Sphere-North-44":"ACAACAACCAGCGAAAGACAGCAGCACCAACCTAA",
"Sphere-South-44":"GAGGAAGGAGACTTTACAAACAAATTATCATCATA",
"Sphere-North-45":"AACGAAAGAGGCACCATGTTACTTAGCC",
"Sphere-South-45":"TTCCTGATTATCACGTAGATTTTCAGGT",
"Sphere-North-46":"GGAACGAGGCGCAGTCAACGGAACAACATTATTACAG",
"Sphere-South-46":"TTAACGTCAGATGAACAAACGAATTTATCAAAATCAT",
"Sphere-North-47":"GTAGAAAGATTCGAGGCATAGTAAGAGCAACACTAATAGTCAGAAGCA",
"Sphere-South-47":"AGGTCTGAGAGAAATCGCAAGACAAAGAACGCGAGACGCCAACATGTA",
"Sphere-North-48":"AAGCGGATTGCAGACCGGAAGCAAACTCCAACAGGTCTATTTTCATTTGG",
"Sphere-South-48":"ATTTAGGCAGAGACAATAAACAACATGTTCAGCTAATATCCGGTATTCTA",
"Sphere-North-49":"CGCAAATGGTCAAAGTTTGAGGGGACGACCAGGTCGACTCTAGACCGCTTT",
"Sphere-South-49":"TAGCAAGCAAATCGCCTAATTTGCCAGTTTTACCGAAGCCCTTTAAGACAC",
"Sphere-North-50":"TGACTATTTCATAACCCTCGTTCTAACGTAACAAAGCTGCTCAT",
"Sphere-South-50":"ATTTAACAAAAACTTTTTCAAAAATAGTATCAAGAAAACAAAAT",
"Sphere-North-51":"TCAGTGAATAAGGCTTGGAAGAAAAATCTAC",
"Sphere-South-51":"TAATTACATTTAACAATAGCTTAGATTAAGA",
"Sphere-North-52":"GTTAATAAAACGAATACCAGACGACGATAAAAACCAATGACCATAAATC",
"Sphere-South-52":"CGCTGAGAAGAGTCTATATTTTAGTTAATTTCATCTTTCAACAGTAGGG",
"Sphere-North-53":"AAAAATCAGGTCCTTTAATTGCTCCTTTTGATAAGAGATTAGATACATTT",
"Sphere-South-53":"CTTAATTGAGAATTATCAACAATAGATAAGTCCTGAATTACCGCGCCCAA",
"Sphere-North-54":"TCTGCGAACGAGCACTCCAGCCAGCTTACGTTGTAAAACGACAATGAATC",
"Sphere-South-54":"GCCGTTTTTATTTCCCAATCCAAATAAAATAATAAGAGCAAGGAAAATTC",
"Sphere-North-55":"AACGAGAAAATAGCGAGAGGCTACGTTGGCCCTGACGAGAAACA",
"Sphere-South-55":"GCCAACGCCTGACCTAAATTTATAGCGATTTCATTTGAATTACC",
"Sphere-North-56":"CCAGAACGAGTATAAATTGTGTCGAAATCCGCGACCTGCTAAAGAATACA",
"Sphere-South-56":"TTTTTTAATGGTTGCACGTAAAACAGAAATAAAGAAATTGGATGATGGCA",
"Sphere-North-57":"CTAAAACACTCATCCGCTTTTGCGGGA",
"Sphere-South-57":"ATTCATCAATATAAAATACATTTGAGG",
"Sphere-North-58":"TCGTCACCCTCAGCATCGCCCACGCATATCCAAAAG",
"Sphere-South-58":"ATTTAGAAGTATTTTATCTAAAATATCTTGAAAAAT",
"Sphere-North-59":"GAGCCTTTTTTTCACGTTGAAAATATAGAAGCGGAGTAACGAT",
"Sphere-South-59":"CTAAAGCACAACAGTGCCACGCTGTAAAACCCAGCCACAGACA",
"Sphere-North-60":"CTAAAGTAAACTACAACGCCTGATAGGTGTATCAC",
"Sphere-South-60":"ATATTTTCAACAGAGATAGAACCTGGTAATATCCA",
"Sphere-North-61":"CGTACTCAGGAGGGATTAGCGGGGTTT",
"Sphere-South-61":"GAACAATATTACGTTGTAGCAATACTT",
"Sphere-North-62":"TGCTCAGTACCAGGAGCGCAGAGGTTGAGGCAGGTC",
"Sphere-South-62":"CTTTGATTAGTAATCTTAATAGCGAAAGGAGCGGGC",
"Sphere-North-63":"AGACGATTGGCCTCTTGAGCCATTTGGGAATTAGAGTGAGGGAGGGAAGG",
"Sphere-South-63":"GCTAGGGCGCTGGGTTGCAGCAAGCGGTCCACGCTGGTTTTTCTTTTCAC",
"Sphere-North-64":"TAAATATTGACAACAAAGTCAGAGGGTAATTGAGCGCCTTTACAGAGAGA",
"Sphere-South-64":"CAGTGAGACGGGCTGGCGAAAGGGGGATGTGCTGCAATTCGCCATTCAGG",
"Sphere-North-65":"ATAACATAAAAATGGGAAGGGCGATCGGTGCGGCCAAGAAC",
"Sphere-South-65":"CTGCGCAACTGTCAGGGAAGCGCATTAGACGGGAAGTACGG",
"Sphere-North-66":"TGTCTGGAAGTTCCAGGCAAAGCGCCAGGCGATTAAGTTGGGTTGGGCG",
"Sphere-South-66":"GGGTATTAAACCAATGAAAATAGCAGCTAATATCAGAGAGATCGACATT",
"Sphere-North-67":"ATTCATTGGGGGTAATAGTAAATGCGATTTGGGCTTGAGATGGT",
"Sphere-South-67":"TTAGTATCATAAATAAGGCGTTCTATTATACATAAATCAATATA",
"Sphere-North-68":"TTAATTTCAACACAAATAAATCCTCATTAA",
"Sphere-South-68":"TGTGAGTGAATTCACGCTGCGCGTAACCAC",
"Sphere-North-69":"AGCCAGAATGGAACGGATAAGTGCCGT",
"Sphere-South-69":"CACACCCGCCGCGAACATCACTTGCCT",
"Sphere-North-70":"CGAGAGGGTAACAAAGTACAACGGAGATT",
"Sphere-South-70":"GAGTAGAAGATGGAAGGGTTAGAACCTAC",
"Sphere-North-71":"TGTATCATCGCCTGAGTAAATTTAAGAACTGGCTCATT",
"Sphere-South-71":"CATATCAAAATTATAAACAGATTAATTTTCCCTTAGAA",
"Sphere-North-72":"ATACCAGTCAGGTTTGCAAAAGAAGTTTTGCCAGAGAATCCCCCTCAAAT",
"Sphere-South-72":"TCCTTGAAAACAATGGTTTGAAATACCGACCGTGTGATATGCGTTATACA",
"Sphere-North-73":"GCTTTAAACAGGGCTTAGAGCTTAATTGCTGAATATAGTTGATTCCCAAT",
"Sphere-South-73":"AATTCTTACCACCCATCCTAATTTACGAGCATGTAGACGAGAACAAGCAA",
"Sphere-North-74":"TATGCAACTAAGAATTAACTGAACACCCTGGGAAATTATT",
"Sphere-South-74":"CCTTATCATTGCCTCTTCGCTATTACGCCAGCAACAGCTG",
"Sphere-North-75":"CATTAAAGGTTCCAATACTGCGGAATCGTATGTTTTAAA",
"Sphere-South-75":"ATTGCCCTTCAATCATAATTACTAGAAAAGGCTGTCTTT",
"Sphere-North-76":"CTGGATAGCGGAATTATCACCGTCACCGATGATATTCACAATTTAATCATT",
"Sphere-South-76":"TAAACACCGGACCGCCTGGCCCTGAGAGACAAGTGTAGCGGAACCTTGCTT",
"Sphere-North-77":"GTGAATTACCTTAATGTTTAGA",
"Sphere-South-77":"CTGTAAATCGTCGAAATAAGAA",
"Sphere-North-78":"CGATTATACCAAGCGCGATGATATAAGTATAGCCCGGATAGCATTCCA",
"Sphere-South-78":"GGATTATACTTCTGAATAAACTCAAACTATCGGCCTTGCCTTCTGACC",
"Sphere-North-79":"CAGACAGCCCGGCTTGCAGGGAGTTAAAGGCTTTGACCCCCAG",
"Sphere-South-79":"TGAAAGCGTAAGATTAGAGCCGTCAATAGATTCCTGATTGTTT",
"Sphere-North-80":"CGGTCGCTGATCATAGTTAGCGTGAGACTCCAAAAAAAAGGCACCGATATATT",
"Sphere-South-80":"AACAACTAATAGAATACGTGGAGAAGAAGAGCCAGCAGCAAATTAGGAGCACT"}

def main():
    matched = False
    numStapsMatched = 0
    
    designs = {}
    designFiles = [f for f in os.listdir(designDir) if not f[0] == '.']
        
    for filename in designFiles:
        s = filename.split('.')
        scaffold = s[0]
        name = '.'.join(s[1:-1])
        designs[name] = scaffold
    
    for design in designs.keys():
        filename = "%s/%s.%s.json" % (designDir, designs[design], design)
        print "\n" + filename
        file = open(filename, 'r')
        lines = file.readlines()
        file.close()
        del file
        
        seqfilename = "%s/%s.txt" % (sequenceDir, designs[design])
        print seqfilename
        seqfile = open(seqfilename, 'r')
        sequence = seqfile.readlines()[0]
        seqfile.close()
        del seqfile
    
        str = ""
        for line in lines:
            str += line
        obj = json.loads(str)
        strands = obj["vstrands"]
        name = obj["name"]
    
        # create dictionaries (keyed by vstrand #) of
        # row/col, scaf array, stap array
        vsNums = []
        vsToScaf = {}
        vsToStap = {}
        vsToLoop = {}
        vsToSkip = {}
        vsToStapColor = {}
    
        for strand in strands:
            num = strand["num"]
            vsNums.append(num)
            row = strand["row"]
            col = strand["col"]
            scaf = strand["scaf"]
            stap = strand["stap"]
            loop = strand["loop"]
            skip = strand["skip"]
            color = strand["stap_colors"]
        
            # DNA sequence will be added in-place to scaf and stap arrays
            # prepopulate index 4 with empty string.
            for i in scaf:
                i.append('')
            for j in stap:
                j.append('')
        
            vsToScaf[num] = scaf
            vsToStap[num] = stap
            vsToLoop[num] = loop
            vsToSkip[num] = skip
            vsToStapColor[num] = color
        
        # extract scaffold 5' breakpoints
        scafBreaks = []
        for vs in vsNums:
            scaf = vsToScaf[vs]
            for i in range(len(scaf)):
                base = scaf[i]
                if (base[1] == -1) & (base[3] != -1):
                    scafBreaks.append([vs, i])
                    initialScafBreak = [vs, i]
        print "scaffold breaks at: "
        print scafBreaks

        # extract staple 5' breakpoints
        stapBreaks = []
        for vs in vsNums:
            stap = vsToStap[vs]
            for i in range(len(stap)):
                base = stap[i]
                if (base[1] == -1) & (base[3] != -1):
                    stapBreaks.append([vs, i])
            
        
        # loop through scaffold permutations,
        # by permuting the sequence string
        
        # start and end permutation indices
        # for when not using custom index set
        startindex = 0
        seqlen = len(sequence)
        endindex = len(sequence)
        
        # initial condition for the sequence as
        # specified in the sequence file
        seq = sequence
                
        print "Running permutations......"
        
        fname = "%s.%s.json" % (designs[design], design)

        indicesToRun = [p for p in range(startindex, endindex)]
        
        for i in indicesToRun:
                        
            seq = sequence
            
            for j in range(i):
                # increment the permutation such that the next
                # 3' end is the current 5' end, i.e., move "forward"
                lastElement = seq[len(seq)-1]
                remaining = seq[0:len(seq)-1]
                seq = lastElement + remaining
            
            # do the computation for a given scaffold break position
            seq2 = seq
            
            # populate scaffold and staple with sequence
            scafBreak = initialScafBreak
            [curr_vs, curr_base] = scafBreak
            [next_vs, next_base] = vsToScaf[curr_vs][curr_base][2:4]
            while next_base != -1:
                n = 1 + vsToLoop[curr_vs][curr_base] \
                + vsToSkip[curr_vs][curr_base]
                # get next n bases
                s = seq2[:n] 
                # throw away used sequence
                seq2 = seq2[n:]
                # append scaffold base(s)
                vsToScaf[curr_vs][curr_base][4] = s 
                # append staple base(s)
                vsToStap[curr_vs][curr_base][4] = comp(s) 
                [curr_vs, curr_base] = [next_vs, next_base]
                [next_vs, next_base] = \
                vsToScaf[curr_vs][curr_base][2:4]
            n = 1 + vsToLoop[curr_vs][curr_base] \
            + vsToSkip[curr_vs][curr_base]
            
            # one more time
            s = seq2[:n] 
            seq2 = seq2[n:] 
            vsToScaf[curr_vs][curr_base][4] = s 
            vsToStap[curr_vs][curr_base][4] = comp(s)
            
            
            # contig stap segments: segments of staple sequence between crossovers
            
            stapContigs = []
                        
            # extract staple sequences and colors, starting at 5' breakpoints
            stapPaths = []
            for stapBreak in stapBreaks:
                color = 0
                path = []
                [curr_vs, curr_base] = stapBreak
                [next_vs, next_base] = vsToStap[curr_vs][curr_base][2:4]
                
                stapContig = ""
                
                while next_base != -1:
                    for stapcolor in vsToStapColor[curr_vs]:
                        if stapcolor[0] == curr_base:
                            color = stapcolor[1]
                    path.append(vsToStap[curr_vs][curr_base][4])
                    stapContig = stapContig + vsToStap[curr_vs][curr_base][4]
                    
                    # if we're at a staple xover
                    if (next_vs != curr_vs):
                        
                        # move on to the next contig
                        stapContigs.append(stapContig)
                        stapContig = ""
                    
                    [curr_vs, curr_base] = [next_vs, next_base]
                    [next_vs, next_base] = vsToStap[curr_vs][curr_base][2:4]
                    
                path.append(vsToStap[curr_vs][curr_base][4])
                stapSeq = ''.join(path)
                
                stapContig = stapContig + vsToStap[curr_vs][curr_base][4]
                stapContigs.append(stapContig)
                stapContig = ""
                                
#                # append staple 3' and 5' modifications according to stap color
#                filename = "%s.%s.json" % (designs[design], design)
#                
#                if append3prime[filename].keys().count(hexToRGB(color)) != 0:
#                    stapSeq = stapSeq + append3prime[filename][hexToRGB(color)]
#                if append5prime[filename].keys().count(hexToRGB(color)) != 0:
#                    stapSeq = append5prime[filename][hexToRGB(color)] + stapSeq
                    
                startpos = "%d[%d]" % (stapBreak[0], stapBreak[1])
                endpos = "%d[%d]" % (curr_vs, curr_base)
                stapPaths.append([startpos, endpos, stapSeq, \
                "%d" % len(stapSeq), hexToRGB(color)])
                
                # check for a match
                matchedThisStaple = False
                for k in staplesToMatch.keys():
                    if staplesToMatch[k] == string.replace(stapSeq, "?", ""):
                        #print "Match at permutation %i" % i
                        #print "Target:\t", staplesToMatch[k], k
                        #print " Value:\t", stapSeq, startpos, endpos
                        matched = True
                        matchedThisStaple = True
                        numStapsMatched += 1
                if matchedThisStaple == False:
                    print "FAILED TO MATCH:\t", stapSeq, startpos, endpos
                    
            if numStapsMatched >= 1:
                print "\nMatched %i staples" % numStapsMatched
                 
            # first line of the staple file gives the 5' end
            # of the scaffold according to the current permutation
            scafBreak1 = initialScafBreak
            [curr_vs1, curr_base1] = scafBreak1
            [next_vs1, next_base1] = vsToScaf[curr_vs1][curr_base1][2:4]
                    
            holdUntil = -1
            for j in range(i):
                # note that vsToSkip is a negative number
                n = 1 - vsToLoop[curr_vs1][curr_base1] \
                - vsToSkip[curr_vs1][curr_base1]
                if n <= 0 and holdUntil == -1: 
                    # get several permutations with the 
                    # same breakpoint position if there is a loop
                    # so don't move the breakpoint yet
                    holdUntil = vsToLoop[curr_vs1][curr_base1]
                    continue
                elif holdUntil > 0:
                    # keep holding on to the breakpoint
                    # as we permute through the loop
                    holdUntil = holdUntil - 1
                    continue
                else: 
                    # finally we're done with the loop
                    # and can move the breakpoint forward
                    holdUntil = -1
                    for m in range(n):
                        if next_base1 == -1:
                            break
                        # if there is a skip, the breakpoint skips
                        # over that position, so we jump it ahead by 2
                        # so move the breakpoint "forward"
                        [curr_vs1, curr_base1] = [next_vs1, next_base1]
                        [next_vs1, next_base1] = vsToScaf[curr_vs1][curr_base1][2:4]
                        
            # if we're back at the beginning earlier than expected,
            # due to skips, or to incomplete use of the scaffold, then we need to quit now
            # because we cannot explicitly model this permutation by
            # a breakpoint shift in caDNAno
            if (i != 0) and (next_base1 == -1) and (i < endindex - 1):
                # we don't want to consider any more permutations,
                # so break the loop through permutations
                print "Permutation loop got back to the original scaffold 5' " \
                + "breakpoint position" \
                + " because the scaffold was used incompletely.\n" \
                + "Consider using all of the scaffold."
                # because we didn't get to endindex, the analysis at the end
                # needs to consider only the permutations we actually did
                endindex = i
                # break out of the for loop
                break
            
            # otherwise, go ahead and record this permutation
            if matched == True:
                print "\nMatch was found at permutation %i, scafBreak:[%04i,%04i]\n" % (i, curr_vs1, curr_base1)
                matched = False
                break
def rev(s):
  return s[::-1]
complement = string.maketrans('ACGTacgt','TGCAtgca')
def comp(s):
  return rev(s.translate(complement))
def hexToRGB(hex):
    r = hex >> 16
    g = (hex ^ hex >> 16 << 16) >> 8
    b = hex >> 8 << 8 ^ hex
    return "#%02x%02x%02x" % (r,g,b)

if __name__ == '__main__':
    main()
