# This module handles parsing of the rotamer library file
#
# Written by YW Huang <whuang@lineartime.com>
# 

"""This module provides a class that represents the rotamer library, which
specifies for each residue all of the rotamers (configuration of sidechains)
that have been identified through statistical methods.

Example:

  >>>import RotamerLib;
  >>>example to follow
"""

import re;

# atomToIdx
atomToIdx = {
		'G':	0,
		'D':	1,
		'E':	2,
		'Z':	3
            };

class RotamerLib:
    """ This class is a container for the rotamer library file.
        It implements methods loadRotamerLibrary and getRotamers
        and contains attributes fileName and rotamers.
    """

    def __init__( self, _fileName = None ):
        self.fileName = _fileName;
        self.rotamers = {};
        if ( _fileName ):
            self.loadRotamerLibrary( _fileName );


    def hasResidue( self, res ):
        return self.rotamers.has_key(res);


    def loadRotamerLibrary( self, _fileName ):
        self.fileName = _fileName;
        self.rotamers = {};

        inp = open( _fileName, 'r' );
        if ( not inp ):
            self.fileName = None;
        else:
            rotamer = re.compile('^\w{3}\s+\d+\s+\d+\s+\d+\s+\d+\s+');
            nonws = re.compile(r'\s+');

            for line in inp.readlines():
                line = line.strip();
                if ( rotamer.match(line) ):
                    data = nonws.split(line);
                    res = data[0];
                    if ( not self.rotamers.has_key(res) ):
                        self.rotamers[res] = [];
                    name = data[0] + data[1] + data[2] + data[3] + data[4];
                    chi = [];
		    if ( len(data) > 11 ):  chi.append( float( data[11] ) );
		    if ( len(data) > 13 ):  chi.append( float( data[13] ) );
		    if ( len(data) > 15 ):  chi.append( float( data[15] ) );
		    if ( len(data) > 17 ):  chi.append( float( data[17] ) );
                    rotentry = {
				'name':		name,
				'chi':		chi
                               };
                    self.rotamers[res].append( rotentry );

            inp.close();


    def numRotamers( self, residue ):
        """ This method returns the number of rotamers defined for
            the specified residue.
        """

        if ( self.rotamers.has_key(residue) ):
            return len( self.rotamers[residue] );

        return 0;


    def getRotamers( self, residue, which = None ):
        """ This method returns the specified rotamer by index position
            for the specified residue.  If no index is provided, all
            rotamers for the specified residue are returned.  You can
            also search by rotamer name by specifying a name for 'which'
        """

        if ( self.rotamers.has_key(residue) ):
            if ( which == None ):
                return self.rotamers[residue];
            elif ( (type(which) == int)  and  (which < len(self.rotamers[residue])) ):
                return self.rotamers[residue][which];
            elif ( type(which) == list ):
                ret = {};
                for idx in which:
                    if ( idx < len(self.rotamers[residue]) ):
                        ret[idx] = self.rotamers[residue][idx];
                    else:
                        ret[idx] = None;

                return ret;
            elif ( type(which) == str ):
                for r in self.rotamers[residue]:
                    if ( r['name'] == which ):
                        return r;

        return None;



    def getDihedral( self, residue, which, atom ):
        """ This method returns the dihedral angle specified by the
            index position for the specified residue and atom name.
        """

        if ( (len(atom) < 2)  or  (not atomToIdx.has_key(atom[1])) ):
             return None;

        idx = atomToIdx[atom[1]];
        r = self.getRotamers( residue, which );
        if ( r  and  (idx < len(r['chi'])) ):
            return r['chi'][idx];

        return None;
