# This module handles parsing of the topology file required to build
# structures from amino acid sequence
#
# Written by YW Huang <whuang@lineartime.com>
# 

"""This module provides a class that represents the residue atom
topology from maintained by the btk project.  This file describes the
build order for atoms in a residue and for each atom provides all angles
and bond lengths required to build the atom forward or backward.

Example:

  >>>from Topology import Topology;
  >>>
  >>>top = Topology( 'top_all27_prot_na.inp' );
  >>>
  >>># find topology line for building HN in MET
  >>>ic = top.findIcTarget( 'MET', 'HN' );
  >>>
  >>># find topology line for building N in ALA in reverse
  >>>ic = top.findIcTarget( 'ALA', 'N', 1 );
"""

import Residue;
import re;

backboneAtoms = [ 'HN', 'N', 'HA', 'CA', 'O', 'C' ];


class Topology:
    """ This class is a container for the BTK structure topology file.
        It implements methods loadTopologyFile and findIcTarget and
        contains attributes fileName, and residues.
    """

    def __init__( self, _fileName = None ):
        self.fileName = _fileName;
        self.residues = {};
        if ( _fileName ):
            self.loadTopologyFile( _fileName );

    def loadTopologyFile( self, _fileName ):
        self.fileName = _fileName;
        self.residues = {};

        inp = open( _fileName, 'r' );
        if ( not inp ):
            self.fileName = None;
        else:
            res = None;
            resi = re.compile('^RESI\s+(\w+)');
            ic = re.compile('^IC\s+');
            atom = re.compile('^ATOM\s+');
            nonws = re.compile(r'\s+');
            r1 = re.compile('^[^H]G1?$');
            r2 = re.compile('^[^H]D1?$');
            r3 = re.compile('^[^H]E1?$');
            r4 = re.compile('^[^H]Z1?$');
            rotRes = {
			'r1': ['A','C','D','E','F','G','H','I','K','L','M','N','P','Q','R','S','T','V','W','Y'],
			'r2': ['R','Q','E','I','L','K','M','N','D','F','Y','H','P','W'],
			'r3': ['R','K','M','Q','E'],
			'r4': ['R','K']
		     };

            for line in inp.readlines():
                line = line.strip();
                m = resi.match(line);
                if ( m ):
                    res = m.group(1);
                    self.residues[res] = { 'charges': {}, 'icdata': [] };
                elif ( res and atom.match(line) ):
                    data = nonws.split(line);
                    self.residues[res]['charges'][data[1]] = float( data[3] );
                elif ( res and ic.match(line) ):
                    data = nonws.split(line);
                    improper = False;
                    v3Name = "";
                    if ( data[3][0] == '*' ):
                        v3Name = data[3][1:];
                        improper = True;
                    else:
                        v3Name = data[3];

                    e = Residue.aaMap.get(res);
                    isRotatable = (
					(r1.match(data[4]) and (e in rotRes['r1'])) or
					(r2.match(data[4]) and (e in rotRes['r2'])) or
					(r3.match(data[4]) and (e in rotRes['r3'])) or
					(r4.match(data[4]) and (e in rotRes['r4']))
				  );

                    icdata = {
        			'v1Name':	data[1],
        			'v2Name':	data[2],
        			'v3Name':	v3Name,
        			'target':	data[4],
        			'len12':	float( data[5] ),
        			'ang123':	float( data[6] ),
        			'dih1234':	float( data[7] ),
        			'ang234':	float( data[8] ),
        			'len34':	float( data[9] ),
				'rotatable':	isRotatable,
				'improper':	improper
                         };
                    self.residues[res]['icdata'].append(icdata);

            inp.close();

    def getTorsionAngles( self, residue ):
        """ This method looks up the dihedral angles from the topology file
            for the specified residue.
        """

        ret = {};
        if ( self.residues.has_key(residue) ):
            ic = self.findIcTarget( residue, '+N' );
            ret['phi'] = ic['dih1234'];
            ic = self.findIcTarget( residue, '+CA' );
            ret['psi'] = ic['dih1234'];
            ic = self.findIcTarget( residue, 'C' );
            ret['omega'] = ic['dih1234'];

        return ret;


    def atomCharge( self, residue, atomName ):
        """ This method finds the charge for the specified atom
            in the specified residue.
        """

        if ( self.residues.has_key(residue)  and  self.residues[residue]['charges'].has_key(atomName) ):
            return self.residues[residue]['charges'][atomName];
        else:
            return 0;


    def findIcTargets( self, residue, inclNext = False ):
        """ This method finds all atoms which are part of the
            specified residue with option to include atoms
            part of the next residue.
        """

        ret = [];
	if ( self.residues.has_key(residue) ):
            if ( inclNext ):
                return self.residues[residue]['icdata'];

            for ic in self.residues[residue]['icdata']:
                if ( not ic['target'][0] == '+' ):
                    ret.append( ic );

        return ret;



    def findNextIcTargets( self, residue ):
        """ This method finds all atoms which are part of the next
            element's backbone.
        """

        ret = [];
        if ( self.residues.has_key(residue) ):
            for ic in self.residues[residue]['icdata']:
                if ( ic['target'][0] == '+' ):
                    ret.append( ic );

        return ret;


    def findIcBackbone( self, residue ):
        """ This method finds all atoms which are part of the
            specified residue's backbone.
        """

        ret = [];
        for atomName in backboneAtoms:
           ret.append( self.findIcTarget( residue, atomName ) );

        return ret;


    def findIcSidechain( self, residue ):
	if self.residues.has_key(residue):
	    return filter( lambda x: x['target'] not in backboneAtoms, self.findIcTargets( residue ) )
	return []


    def findIcTarget( self, residue, target, rev = 0 ):
        """ This method searches the topology file for the specified residue
            and returns the data from the ic line corresponding to the
            specified target atom.  Specifying rev = 1 will give you the ic
            lines for building the target atom in reverse order.
        """

        if ( self.residues.has_key(residue) ):
            for ic in self.residues[residue]['icdata']:
                if ( ( rev == 0 ) and ( ic['target'] == target ) ):
                    return ic;
                elif ( ( rev != 0 ) and ( ic['v1Name'] == target ) ):
                    return ic;

        return None;
