# This module contains some useful functions for building
# protein structures
#
# Written by YW Huang <whuang@lineartime.com>
# 

"""This module contains functions for building structures

Example:

  >>>import Structure;
  >>>
  >>>need example
"""

import re;
import Residue;

from math import *;
from Vector import Vector;


def setVectorFromDihedral(v1, v2, v3, len34, ang234, dih1234):
    """ This method takes 3 depedent atoms and returns vector of target
        atom's location given bond length, angle and dihedral.  This is
        based on btk's function.
    """

    z = v3 - v2;
    z /= z.length();
    y = z.cross( v1 - v2 );
    y /= y.length();
    x = y.cross(z);

    return v3 + (len34*((x*cos(radians(dih1234)) + y*sin(radians(dih1234)))*sin(radians(ang234))
                        - z*cos(radians(ang234))));


def vectorDihedral(v1, v2, v3, v4):
    """ This method takes 4 vectors and computes the dihedral angle.
    """

    b_v1 = v1 - v2;
    b_v2 = v3 - v2;
    b_v3 = v4 - v3;

    yaxis = b_v2.cross(b_v1).normal();
    xaxis = yaxis.cross(b_v2).normal();

    return degrees( atan2( b_v3.inner_prod(yaxis), b_v3.inner_prod(xaxis) ) );


def getChiAngles( top, res, atoms ):
    """ This method takes a hash of atoms constituting a single residue,
        and determined the chi angles using the topology data struct.
    """

    chi = [];
    r1 = re.compile('^[^H]G1?$');
    r2 = re.compile('^[^H]D1?$');
    r3 = re.compile('^[^H]E1?$');
    r4 = re.compile('^[^H]Z1?$');
    rotRes = {
		'r1': ['A','C','D','E','F','G','H','I','K','L','M','N','P','Q','R','S','T','V','W','Y'],
		'r2': ['R','Q','E','I','L','K','M','N','D','F','Y','H','P','W'],
		'r3': ['R','K','M','Q','E'],
		'r4': ['R','K']
             };

    e = Residue.aaMap[res];
    for ic in top.findIcTargets( res ):
        atomName = ic['target'];
        isRotatable = (
			(r1.match(atomName) and (e in rotRes['r1'])) or
			(r2.match(atomName) and (e in rotRes['r2'])) or
			(r3.match(atomName) and (e in rotRes['r3'])) or
			(r4.match(atomName) and (e in rotRes['r4']))
		      );

        if ( isRotatable  and  not ic['improper'] ):
            v1 = atoms[ ic['v1Name'] ]['pos'];
            v2 = atoms[ ic['v2Name'] ]['pos'];
            v3 = atoms[ ic['v3Name'] ]['pos'];
            v4 = atoms[ ic['target'] ]['pos'];

            chi.append( vectorDihedral( v1, v2, v3, v4 ) );

    return chi;
