# This module contains all Energy-related functions, however to calculate
# energy, one should use the methods associated with class Rotamer.
#
# Written by YW Huang <whuang@lineartime.com>
#

"""This module contains all Energy-related functions.  At present,
LJ and naive/sternberg dielectric functions are supported.  LJ parameters
must be read and parsed from the lj_params.inp file.

Example:

  >>>import Energy;
  >>>examples go here
"""

import math;
import re;

# constants
c_c = 299792458;
c_m0 = 4 * math.pi * 10**-7;
c_e0 = 1 / (c_c**2 * c_m0);
c_k = 1 / (4 * math.pi * c_e0);


# it shouldn't go here
def atomSqrDistance(a, b):
    """ This computes the square of the distance between atoms a and b.
    """
    x1 = a['pos'].x();
    y1 = a['pos'].y();
    z1 = a['pos'].z();
    x2 = b['pos'].x();
    y2 = b['pos'].y();
    z2 = b['pos'].z();

    return (x2-x1)**2 + (y2-y1)**2 + (z2 - z1)**2

def atomDistance(a, b):
    """ This computes the distance between atoms a and b.
    """

    x1 = a['pos'].x();
    y1 = a['pos'].y();
    z1 = a['pos'].z();
    x2 = b['pos'].x();
    y2 = b['pos'].y();
    z2 = b['pos'].z();

    return  math.sqrt( (x2-x1)**2 + (y2-y1)**2 + (z2 - z1)**2 );


def genericDielectric(a1, a2, epsfunc):
    r = atomDistance(a1, a2);
    ep = epsfunc(r);
    pairEnergy = 332 * a1['charge'] * a2['charge'] / (ep * r);
    return pairEnergy;


def naiveEps(r):
    return 6 * r;

def naiveDielectric(a1, a2, lj = None):
    return genericDielectric(a1, a2, naiveEps);

def sternbergEps(r):
    return (((r<6)*4) + ((r>6 and r<8)*(38*r-224)) + ((r>8)*80));

def sternbergDielectric(a1, a2, lj = None):
    return genericDielectric(a1, a2, sternbergEps);

def pairEnergy(a1, a2, energyFunc, lj = None):
    """ This is the generic pair energy function wrapper.  Specify atoms
        a1 and a2, and the energyFunc to use to compute the pair energy.
        Supported functions are "naiveDielectric", "sternbergDielectric",
        and "lennardJones".  For the latter, one must also pass in the
        LJ parameter data structure obtained by calling loadLJParameters.
    """

    return energyFunc(a1, a2, lj);    

def chainEnergy(atoms1, atoms2, energyFunc, lj = None):
    """ This function computes the energy between two lists of atoms
        using the specified energy function.
    """

    totalEnergy = 0;
    for a1 in atoms1:
        for a2 in atoms2:
            totalEnergy = totalEnergy + pairEnergy(a1, a2, energyFunc, lj);

    return totalEnergy;


def lennardJones(a, b, lj, maxValue = 10.0):
    """ This method computes the LJ energy between two atoms a and b.
    """

    if ( lj == None ):
        print "Error: You must pass in the LJ parameter data structure when calling pairEnergy using lennardJones";
        return 0;

    r = atomDistance(a, b);
    Eps_i_j = math.sqrt( lj[a['name'][0]]['epsilon'] * lj[b['name'][0]]['epsilon'] );
    Rmin_i_j = lj[a['name'][0]]['Rmin_2'] + lj[a['name'][0]]['Rmin_2'];
    V = Eps_i_j * ( (Rmin_i_j/r)**12 - 2*(Rmin_i_j/r)**6 );
    if V > maxValue:
	V = maxValue;
    return V;

def loadLJParameters(filename):
    """ This method opens and parses the LJ parameters file and returns a
        a data structure which maps atom names to LJ parameters epsilon,
        Rmin_2.
    """

    lj = {};
    inp = open(filename, 'r');

    comment = re.compile('^\s*!');
    nonws = re.compile(r'\s+');

    for line in inp.readlines():
        line = line.strip();
        if ( not comment.match(line)  and  (len(line) > 0) ):
            data = nonws.split(line);
            atomName = data[0];
            epsilon = float( data[2] );
            Rmin_2 = float( data[3] );
            if ( lj.has_key(atomName) ):
                print "Warning: Duplicate atom " + atomName + " in '" + filename + "' has been ignored.";
            else:
                lj[atomName] = { 'epsilon': epsilon, 'Rmin_2': Rmin_2 };

    inp.close();
    return lj;

