cims = function(n.reps=8, sd2)
{
	setupClust()

	gpars = cbind(runif(n.reps, 0, 100), runif(n.reps, 0, 50), sd2)
	res = apply(gpars, 1, with_counter(with_timer(optim.cims), n.reps)) 

	idx = which.min(sapply(res, function(x) x$nLL))
	best = res[[idx]]

	best$df = .k+1 + 2

	class(best) = 'cims'
	
	invisible(best)
}

lines.cims = function(x, ..., mean=T, col='orange', lwd=3)
{
	p = predict(x)
	
	lines()
}

cims.subj_only = function(gpars)
{
	setupClust()
	sbj = t(array(parRapply(cl, count_mat, optim.subj, gpars=gpars), c(.k+2, nrow(count_mat))))

	cm = list()
	cm$spars = sbj
	cm$gpars = gpars
	
	cm$df = .k+1
	
	class(cm) = 'cims'
	
	return(cm)
}

load_cims_model = function(file)
{
	d = read.csv(file, header=T, row.names=1)
	
	# number of conditions is #col minus 3 gpars, nLL, and pc1
	k = ncol(d) - 5
		
	i = k + 3

	cm = list()
	cm$gpars = as.numeric(d[1,i:(i+2)])
	cm$spars = as.matrix(d[,1:(k+2)])
	
	class(cm) = 'cims'
	
	return(cm)
}


as.data.frame.cims = function(x, ...)
{
	df = data.frame(x$spars)
	colnames(df) = c('subj_pc1', paste0('subj_sd', 1:.k), 'nLL')
	
	df$gpar_mc1 = x$gpars[1]
	df$gpar_sdc1 = x$gpars[2]
	df$gpar_sdc2 = x$gpars[3]
	
	row.names(df) = row.names(count_mat)

	return(df)
}

optim.cims = function(gpars, n.steps=6)
{
	nLL = Inf
	for (step in 1:n.steps) {
		spars = t(array(parRapply(cl, count_mat, optim.subj, gpars=gpars), c(.k+2, nrow(count_mat))))

		#this call takes ~ 30s on 8 threads
		#h = optim.time(c(gpars[1], gpars[2]), negLL.hpar, spar=spar, sd2=gpars[3], count_mat=count_mat)

		#let's optimize each par separately
		#takes ~ 3s each on 8 threads
		h.mu  = optimize(negLL.gpar, c(-300, 300), sd1=gpars[2], sd2=gpars[3], spars=spars, count_mat=count_mat)		
		h.sd1 = optimize(negLL.gpar, c(0, gpars[3]-1), mu=h.mu$minimum, sd2=gpars[3], spars=spars, count_mat=count_mat)

		gpars[1:2] = c(h.mu$minimum, h.sd1$minimum)
		
		cat('\tStep', step, 'of',  n.steps, h.sd1$objective, '\n')
		
		if (h.sd1$objective < nLL) {
			nLL = h.sd1$objective
			best.spars = spars
			best.gpars = gpars
		}
	}
	return (list(nLL=nLL, spars=best.spars, gpars=best.gpars))
}

optim.subj = function(y, gpars, n.steps=25, p.seqlen=4)
{
	# because this method is getting called in its own thread
	# source the file to ensure the relevant methods are in scope
	source('cims.R')

	nLL = Inf
	for(p1 in seq(.01, .99, length.out = p.seqlen)) {
		sds = rep(0,.k)
		for(i in 1:n.steps) {

			sds = apply(.idx, 2, function(i) optimize(negLL.cims, c(1,1e3), p1=p1, gpars=gpars, y=y[i])$minimum)

			h = optimize(negLL.subj, c(1e-6, 1 - 1e-6), sds=sds, gpars=gpars, y=y)	
			p1 = h$minimum

			if (h$objective < nLL) {
				nLL = h$objective
				pars = c(p1, sds)
			}
		}
	}

	return (c(pars, nLL))	
}

negLL.gpar.parallel = function(i, spars, gpars)
{
	source("cims.R")
	
	negLL.subj(spars[i, 2:(.k+1)], spars[i,1], gpars, count_mat[i,])
}

negLL.gpar = function(mu, sd1, sd2, spars, count_mat)
{
	sum(parSapply(cl, seq(n.sub), negLL.gpar.parallel, spars=spars, gpars=c(mu, sd1, sd2)))
}

negLL.subj = function(sds, p1, gpars, y)
{
	sum(sapply(seq(.k), function(i) negLL.cims(sds[i], p1, gpars, y[.idx[,i]])))
}

negLL.cims = function(sd, p1, gpars, y)
{
	p = pred.cims(asyncs, p1, gpars[1], gpars[3], sd, gpars[2])
	ll = LL.binomial(p, y, max_count)

	return (-sum(ll))
}

logLik.cims = function(object, ..., separate=F)
{
	cims.p = predict(object)

	#calc ll for each condition		
	ll = apply(.idx, 2, function(i) rowSums(LL.binomial(cims.p[,i], count_mat[,i], max_count)))

	if (!separate){
		ll = rowSums(ll)

		n = max_count*ncol(count_mat)

		#let's check if the df is already stored	
		if(any(names(object) == 'df')) {
			df = object$df
		} else {
			# we take one off b/c sd.c2 is fixed
			n.hpar = length(object$gpars) - 1

			#we take one off b/c last column is nLL
			n.spar = ncol(object$spars)-1
			
			df = n.spar + n.hpar/n.sub
		}

		attributes(ll) = list(df=df, nobs=n, nall=n)
	
		class(ll) = 'logLik'
	}

	return (ll)
}


as.var = function(sd)
{
	if (sd < 1e-10)
		return(1e-10)

	return (sd**2)
}

pred.cims = function (async, p.c1, mu.c1, sd.2, sd.x, sd.1)
{
	#safely transform sd into variance
	var.1 = as.var(sd.1)
	var.2 = as.var(sd.2)
	var.x = as.var(sd.x)

	p.c1 = clip(p.c1)
	
	lprior = 2 * log(p.c1 / (1-p.c1))
	
	b = log((var.x + var.2) / (var.x + var.1)) +  (mu.c1**2 / (var.2 - var.1))
	
	c = (1 / (var.x + var.1)) - (1 / (var.x + var.2)) 

	if (lprior < -b) {
		lprior = -b
	}

	bound = sqrt((lprior+b)/c)
	middle = mu.c1 * (var.x+var.2)/(var.2-var.1)

	upper = middle + bound
	lower = middle - bound

	p.cdf = pnorm(upper, async, sd.x) - pnorm(lower, async, sd.x)

	return (p.cdf)
}

predict.cims.par = function(pars)
{
	return(pred.cims(asyncs, pars[1], pars[2], pars[4], pars[5], pars[3]))
}

predict.cims = function(object, ...)
{
	spars = object$spars
	gpars = object$gpars
	
	sds = object$spar[,2:(.k+1)]

	pred = array(0, c(nrow(count_mat), .k*length(asyncs)))
	for(i in 1:.k) {
		pars = cbind(spars[,1], t(array(gpars[1:3], c(3, nrow(count_mat)))), sds[,i])
		pred[,.idx[,i]] = t(apply(pars, 1, predict.cims.par))
	}

	return(pred)
}

