function [results] = param_sensitivities()


    %% Model settings

    model.AbsoluteTolerance = 1e-7;
    model.RelativeTolerance = 1e-5;

    model.BNGL_fname = 'YMC';
    model.BNG2_path = 'C:\BioNetGen_2.0.46\Perl2\BNG2.pl';

        %a factor to convert #/cell to uM (and vice versa)
        %this factor assumes a cell volume of 29fL (as does the model)
    num2uM = 1e6/(29e-15 * 6.022e23);


    %% Define the observables to be used in the model

    obs{1}.name = 'Fus3pTpY';
    obs{1}.observables = {'Fus3pTpY'};
    obs{1}.fit_type = 'scale no shift';  %just scale the data, do not apply a baseline shift
    obs{1}.data_scale.value = ['return_abund(new_params,''Fus3_num'') * ' num2str(num2uM) ' * 0.5']; %start with scale value=0.5*total_Fus3_concentration
    obs{1}.data_scale.lb = ['return_abund(new_params,''Fus3_num'') * ' num2str(num2uM) ' * 0.33'];   %lower bound of scale value=0.33*total_Fus3_concentration
    obs{1}.data_scale.ub = ['return_abund(new_params,''Fus3_num'') * ' num2str(num2uM) ' * 0.67'];   %upper bound of scale value=0.67*total_Fus3_concentration

    obs{2}.name = 'Ste5 at membrane';
    obs{2}.observables = {'Ste4_Ste5'};
    obs{2}.fit_type = 'scale and shift'; %apply baseline shift AND scale to the data
    obs{2}.data_scale.value = ['return_abund(new_params,''Ste5_num'') * ' num2str(num2uM) ' * 0.5']; %start with scale value=0.5*total_Ste5_concentration
    obs{2}.data_scale.lb = ['return_abund(new_params,''Ste5_num'') * ' num2str(num2uM) ' * 0.2'];    %lower bound of scale value=0.2*total_Ste5_concentration
    obs{2}.data_scale.ub = ['return_abund(new_params,''Ste5_num'') * ' num2str(num2uM)];             %upper bound of scale value=total_Ste5_concentration
    obs{2}.data_shift.value = ['return_abund(new_params,''Ste5_num'') * ' num2str(num2uM) ' * 0.01']; %start with shift value=0.01*total_Ste5_concentration
    obs{2}.data_shift.lb = ['0'];                                                                     %lower bound of shift value=0
    obs{2}.data_shift.ub = ['return_abund(new_params,''Ste5_num'') * ' num2str(num2uM) ' * 0.02'];    %upper bound of shift value=0.02*total_Ste5_concentration


    %% Initialize the experiments
    %The first index of 'expt' is the experiment number, representing different
    %physical experiments.  The second index of 'expt' is the part of the
    %experiment, representing different experimental conditions (for example,
    %steady state is part 1, and pheromone added for part 2).  For each part,
    %we can specify which model species (expt{}{}.species) or parameters
    %(expt{}{}.param) we want to change.  For each part we can also specify
    %observed experimental results (expt{}{}.obs{}) which correspond to
    %observables in the model (for example, 'Fus3pTpY').  The error of the
    %observed experimental results can also be given (for later plotting
    %purposes).


    %expt 1 - simulate to steady state
    expt{1}{1}.name = 'To steady state';
    expt{1}{1}.duration = 100000;
    expt{1}{1}.species.names = {};
    expt{1}{1}.species.values = [];
    expt{1}{1}.param.names = {'kbind_Gpa1_Ste4','koff_Gpa1_Ste4'};  %assign these model parameters to be equal to the listed parameters on the next line
    expt{1}{1}.param.values = {'kbind_Gpa1_Ste4_0nM','koff_Gpa1_Ste4_0nM'};
    expt{1}{1}.num_t = 10;
    expt{1}{1}.obs = {};

    %expt 1 - Once at SS, add 100nM pheromone
    expt{1}{2}.name = '100 nM pheromone';
    expt{1}{2}.duration = 1000;
    expt{1}{2}.num_t = 500;
    expt{1}{2}.species.names = {};
    expt{1}{2}.species.values = [];
    expt{1}{2}.param.names = {'kbind_Gpa1_Ste4', 'koff_Gpa1_Ste4'}; 
    expt{1}{2}.param.values = {'kbind_Gpa1_Ste4_100nM','koff_Gpa1_Ste4_100nM'};
    expt{1}{2}.obs{1}.name = 'Fus3pTpY';
    expt{1}{2}.obs{1}.t_data = [0, 1, 1.5, 2.5, 3.5, 5, 7, 10, 12.5, 15]*60; %time points in seconds
    expt{1}{2}.obs{1}.data = [0.11345455, ...                                %corresponding data, normalized to max of 1
                0.37164963, ...
                0.57861288, ...
                0.76623377, ...
                0.8378005, ...
                0.97071014, ...
                0.91434098, ...
                0.96546007, ...
                0.9231832, ...
                1];
    expt{1}{2}.obs{1}.data_error = [0.045876762, ...                        %corresponding data error (standard error)
                0.018061813, ...
                0.041151423, ...
                0.03643465, ...
                0.019224123, ...
                0.023261757, ...
                0.057334347, ...
                0.026576762, ...
                0.019981266, ...
                0.071773418, ...
                ];          
    expt{1}{2}.obs{2}.name = 'Ste5 at membrane';
    expt{1}{2}.obs{2}.t_data = [0 6 12 18 24 30 36 42 48 54 60 66 72 78 84 90 ...
                                 96 102 108 114 120 126 132 138 144 150 156 162 168 174];
    expt{1}{2}.obs{2}.data = [ 0, ...
                0.284204966, ...
                0.438457475, ...
                0.528262018, ...
                0.820390914, ...
                0.47649234, ...
                0.771790808, ...
                0.746434231, ...
                0.683042789, ...
                0.916534601, ...
                0.855256207, ...
                0.89276281, ...
                0.941362916, ...
                0.963021659, ...
                0.937665082, ...
                0.980982567, ...
                0.831484416, ...
                0.854199683, ...
                0.863180137, ...
                0.833597464, ...
                0.875330164, ...
                0.779186476, ...
                0.970945589, ...
                0.973058637, ...
                0.751716852, ...
                0.743264659, ...
                0.861067089, ...
                0.849973587, ...
                1, ...
                0.866877971, ...
                ];
    expt{1}{2}.obs{2}.data_error = [0, ...
                0.141448494, ...
                0.085482937, ...
                0.058799049, ...
                0.099179186, ...
                0.097053883, ...
                0.072495193, ...
                0.07414823, ...
                0.055729213, ...
                0.103429688, ...
                0.133419968, ...
                0.097526149, ...
                0.075328896, ...
                0.082176968, ...
                0.089969678, ...
                0.104374221, ...
                0.040380137, ...
                0.08005177, ...
                0.0774542, ...
                0.121376651, ...
                0.049825779, ...
                0.210401479, ...
                0.149241416, ...
                0.157977813, ...
                0.085482937, ...
                0.104374221, ...
                0.141920761, ...
                0.106972002, ...
                0.181356577, ...
                0.099651453, ...
                ];

    %expt 2 - simulate to steady state
    expt{2}{1}.name = 'To steady state';
    expt{2}{1}.duration = 100000;
    expt{2}{1}.species.names = {};
    expt{2}{1}.species.values = [];
    expt{2}{1}.param.names = {'kbind_Gpa1_Ste4','koff_Gpa1_Ste4'};
    expt{2}{1}.param.values = {'kbind_Gpa1_Ste4_0nM','koff_Gpa1_Ste4_0nM'};
    expt{2}{1}.num_t = 10;
    expt{2}{1}.obs = {};

    %expt 2 - Once at SS, add 0.1nM pheromone
    expt{2}{2}.name = '0.1 nM pheromone';
    expt{2}{2}.duration = 1000;
    expt{2}{2}.num_t = 500;
    expt{2}{2}.species.names = {};
    expt{2}{2}.species.values = [];
    expt{2}{2}.param.names = {'kbind_Gpa1_Ste4', 'koff_Gpa1_Ste4'};
    expt{2}{2}.param.values = {'kbind_Gpa1_Ste4_0_1nM','koff_Gpa1_Ste4_0_1nM'};
    expt{2}{2}.obs{1}.name = 'Fus3pTpY';
    expt{2}{2}.obs{1}.t_data = [0, 1, 1.5, 2.5, 3.5, 5, 7, 10, 15]*60;
    expt{2}{2}.obs{1}.data = [0.176834852, ...
                0.329371139, ...
                0.288012747, ...
                0.558704616, ...
                0.687680218, ...
                0.624247385, ...
                0.577729277, ...
                0.665405012, ...
                0.624194435];
    expt{2}{2}.obs{1}.data_error = [0.038525552, ...
                0.288575862, ...
                0.138961848, ...
                0.171095673, ...
                0.152591642, ...
                0.00214348, ...
                0.082528078, ...
                0.025162381, ...
                0.034410183, ...
                ];          
    expt{2}{2}.obs{2}.name = 'Ste5 at membrane';
    expt{2}{2}.obs{2}.t_data = [0 6 12 18 24 30 36 42 48 54 60 66 72 78 84 90 ...
                                 96 102 108 114 120 126 132 138 144 150 156 162 168 174];
    expt{2}{2}.obs{2}.data = [0, ...
                0.021130481, ...
                0.068674062, ...
                0.083465399, ...
                0.096143687, ...
                0.075013207, ...
                0.063391442, ...
                0.046487058, ...
                0.105652404, ...
                0.132065504, ...
                0.019017433, ...
                0.071843634, ...
                0.034865293, ...
                0.061278394, ...
                0.080295827, ...
                0.051769678, ...
                0.048600106, ...
                0.025356577, ...
                0.073956683, ...
                0.087691495, ...
                0.196513471, ...
                0.140517697, ...
                0.141574221, ...
                0.116217644, ...
                0.165874274, ...
                0.117274168, ...
                0.072900158, ...
                0.099313259, ...
                0.150026413, ...
                0.142630745, ...
                ];
    expt{2}{2}.obs{2}.data_error = [0, ...
                0.038482409, ...
                0.042330586, ...
                0.02338542, ...
                0.051211199, ...
                0.026049551, ...
                0.044106709, ...
                0.030785843, ...
                0.043810671, ...
                0.039074379, ...
                0.046474908, ...
                0.043810671, ...
                0.028121712, ...
                0.035818172, ...
                0.054467406, ...
                0.049731009, ...
                0.034930164, ...
                0.064235922, ...
                0.043810671, ...
                0.040554464, ...
                0.076372742, ...
                0.052395246, ...
                0.056243423, ...
                0.048250924, ...
                0.027825674, ...
                0.048250924, ...
                0.064235922, ...
                0.040258426, ...
                0.04677084, ...
                0.044994717, ...
                ];



    %Pair up the observables from obs{} with the observables from
    %expt{}{}.obs{}.
    for expt_num = 1:length(expt)
        for expt_part = 1:length(expt{expt_num})
            for expt_obs_num = 1:length(expt{expt_num}{expt_part}.obs)
                for obs_num = 1:length(obs)
                    if isequal(expt{expt_num}{expt_part}.obs{expt_obs_num}.name,obs{obs_num}.name)
                        expt{expt_num}{expt_part}.obs{expt_obs_num}.obs_index = obs_num;
                    end
                end
            end
        end
    end


    %% Initialize the model

    %convert the bngl file into a network file.
    [s, w] = system([model.BNG2_path ' ' model.BNGL_fname '.bngl']);

    params_for_sensitivity(1).name = 'Fus3_num';
    params_for_sensitivity(1).value = 20400;
    params_for_sensitivity(end+1).name = 'kbind_Gpa1_Ste4_0_1nM';
    params_for_sensitivity(end).value = 0.37091;
    params_for_sensitivity(end+1).name = 'kbind_Gpa1_Ste4_0nM';
    params_for_sensitivity(end).value = 0.10273;
    params_for_sensitivity(end+1).name = 'kbind_Gpa1_Ste4_100nM';
    params_for_sensitivity(end).value = 0.014296;
    params_for_sensitivity(end+1).name = 'kcat_Msg5_MAPK_PO4';
    params_for_sensitivity(end).value = 0.062661;
    params_for_sensitivity(end+1).name = 'kcat_nonspecific_dephosph';
    params_for_sensitivity(end).value = 0.0021919;
    params_for_sensitivity(end+1).name = 'kcat_Ptp_MAPK_PO4';
    params_for_sensitivity(end).value = 2.8774;
    params_for_sensitivity(end+1).name = 'kcat_Ste11pSpSpTSte5Ste7_pS';
    params_for_sensitivity(end).value = 3.6382;
    params_for_sensitivity(end+1).name = 'kcat_Ste11pSpSpTSte5Ste7pS_pT';
    params_for_sensitivity(end).value = 4.5313;
    params_for_sensitivity(end+1).name = 'kcat_Ste4Ste18Ste5Ste11_pS';
    params_for_sensitivity(end).value = 2.7582;
    params_for_sensitivity(end+1).name = 'kcat_Ste5Ste7pSpTFus3_pT';
    params_for_sensitivity(end).value = 0.70423;
    params_for_sensitivity(end+1).name = 'kcat_Ste5Ste7pSpTFus3_pY';
    params_for_sensitivity(end).value = 4.4084;
    params_for_sensitivity(end+1).name = 'kcat_Ste5Ste7pSpTFus3pT_pY';
    params_for_sensitivity(end).value = 0.48499;
    params_for_sensitivity(end+1).name = 'kcat_Ste5Ste7pSpTFus3pY_pT';
    params_for_sensitivity(end).value = 5.051;
    params_for_sensitivity(end+1).name = 'Kd_Msg5_MAPK';
    params_for_sensitivity(end).value = 1.4;
    params_for_sensitivity(end+1).name = 'Kd_Ste5_Fus3';
    params_for_sensitivity(end).value = 1;
    params_for_sensitivity(end+1).name = 'Kd_Ste5_Ste11';
    params_for_sensitivity(end).value = 0.140124976;
    params_for_sensitivity(end+1).name = 'Kd_Ste5_Ste7';
    params_for_sensitivity(end).value = 0.001952477;
    params_for_sensitivity(end+1).name = 'Kd_Ste7_MAPK';
    params_for_sensitivity(end).value = 0.1;
    params_for_sensitivity(end+1).name = 'kdilution';
    params_for_sensitivity(end).value = 0.000128;
    params_for_sensitivity(end+1).name = 'koff_Gpa1_Ste4_0_1nM';
    params_for_sensitivity(end).value = 0.034849;
    params_for_sensitivity(end+1).name = 'koff_Gpa1_Ste4_0nM';
    params_for_sensitivity(end).value = 0.00039613;
    params_for_sensitivity(end+1).name = 'koff_Gpa1_Ste4_100nM';
    params_for_sensitivity(end).value = 0.048251;
    params_for_sensitivity(end+1).name = 'koff_Ptp_MAPK';
    params_for_sensitivity(end).value = 0.064643;
    params_for_sensitivity(end+1).name = 'koff_Ste4Ste18_Ste5';
    params_for_sensitivity(end).value = 0.26373;
    params_for_sensitivity(end+1).name = 'koff_Ste5_Fus3';
    params_for_sensitivity(end).value = 2.3;
    params_for_sensitivity(end+1).name = 'kon_Msg5_MAPK';
    params_for_sensitivity(end).value = 0.92293;
    params_for_sensitivity(end+1).name = 'kon_Ptp_MAPK';
    params_for_sensitivity(end).value = 1.0082;
    params_for_sensitivity(end+1).name = 'kon_Ste4Ste18_Ste5';
    params_for_sensitivity(end).value = 1.8799;
    params_for_sensitivity(end+1).name = 'kon_Ste5_Ste11';
    params_for_sensitivity(end).value = 0.10562;
    params_for_sensitivity(end+1).name = 'kon_Ste5_Ste7';
    params_for_sensitivity(end).value = 3.775;
    params_for_sensitivity(end+1).name = 'kon_Ste7_MAPK';
    params_for_sensitivity(end).value = 3.5466;
    params_for_sensitivity(end+1).name = 'Msg5_num';
    params_for_sensitivity(end).value = 38;
    params_for_sensitivity(end+1).name = 'Ptp_num';
    params_for_sensitivity(end).value = 193.6644;
    params_for_sensitivity(end+1).name = 'Ste11_num';
    params_for_sensitivity(end).value = 3500;
    params_for_sensitivity(end+1).name = 'Ste11_pS_only_PO4_factor';
    params_for_sensitivity(end).value = 2089.636;
    params_for_sensitivity(end+1).name = 'Ste4_num';
    params_for_sensitivity(end).value = 818;
    params_for_sensitivity(end+1).name = 'Ste5_num';
    params_for_sensitivity(end).value = 480;
    params_for_sensitivity(end+1).name = 'Ste7_num';
    params_for_sensitivity(end).value = 920;
    params_for_sensitivity(end+1).name = 'Ste7_pS_only_PO4_factor';
    params_for_sensitivity(end).value = 3871.1816;
    params_for_sensitivity(end+1).name = 'Cell_volume';
    params_for_sensitivity(end).value = 2.9E-14;


    for param_val_num = 1:length(params_for_sensitivity)

        new_params={};

        %Compile BNGL commands to set the estimated parameters 
        for j = 1:length(params_for_sensitivity)
            if j~= param_val_num
                new_params{end+1} = ['setParameter("' char(params_for_sensitivity(j).name) '","' ...
                         num2str(params_for_sensitivity(j).value) '")'];
            end
        end

        %add the BNGL command to set the parameter whose sensitivity we are measuring
        new_params{end+1} = ['setParameter("' params_for_sensitivity(param_val_num).name '","' ...
                         num2str(params_for_sensitivity(param_val_num).value * 0.95) '")'];

        model.fname = ['param_sens_' num2str(param_val_num) '_low'];             
        %run the simulation
        write_and_execute_bngl_file(model, expt, obs, new_params);


        %change the BNGL command to set the parameter whose sensitivity we are measuring
        new_params{end} = ['setParameter("' params_for_sensitivity(param_val_num).name '","' ...
                         num2str(params_for_sensitivity(param_val_num).value * 1.05) '")'];

        model.fname = ['param_sens_' num2str(param_val_num) '_high'];             
        %run the simulation
        write_and_execute_bngl_file(model, expt, obs, new_params);

    end




% RUN THE SIMULATIONS ELSEWHERE BECAUSE MATLAB CAN'T TALK TO BNG IN
% WINDOWS



    results{1,1} = 'parameter';
    results{1,2} = 'delta(dynamic range)';
    results{1,3} = 'delta(pathway output)';
    
    for param_val_num = 1:length(params_for_sensitivity)

        model.fname = ['param_sens_' num2str(param_val_num) '_low'];
        [sim_low expt] = read_sim_results(model, expt, obs);        
        model.fname = ['param_sens_' num2str(param_val_num) '_high'];             
        [sim_high expt] = read_sim_results(model, expt, obs);        

        %get the induced and basal pathway output
        for expt_num = 1:length(sim_low)
            if isequal(expt{expt_num}{2}.name, '100 nM pheromone')
                data_index = expt{expt_num}{2}.obs{1}.data_index;
                sig_low = sim_low{expt_num}{2}.Data(end,data_index);
                sig_high = sim_high{expt_num}{2}.Data(end,data_index);
            end
        end
        basal_sig_low = sim_low{1}{1}.Data(end,data_index);
        basal_sig_high = sim_high{1}{1}.Data(end,data_index);

        delta_dynamic_range(param_val_num) = (sig_high / basal_sig_high) - (sig_low / basal_sig_low);
        delta_pathway_output(param_val_num) = (sig_high - sig_low);

        results{param_val_num+1,1} = params_for_sensitivity(param_val_num).name;
        results{param_val_num+1,2} = delta_dynamic_range(param_val_num);
        results{param_val_num+1,3} = delta_pathway_output(param_val_num);
    end

    
    

    
    % Now investigate how overexpression of Ste5 affects Ste7 sensitivity
    
    Ste5_num_index = 38;
    Ste7_num_index = 39;
    %Compile BNGL commands to set the estimated parameters 
    new_params={};
    for j = 1:length(params_for_sensitivity)
        new_params{end+1} = ['setParameter("' char(params_for_sensitivity(j).name) '","' ...
                 num2str(params_for_sensitivity(j).value) '")'];
    end
    model.fname = ['WT'];             
    write_and_execute_bngl_file(model, expt, obs, new_params);

    %add the BNGL command to increase the Ste7_num by 10x
    new_params{end+1} = ['setParameter("Ste7_num","' ...
                     num2str(params_for_sensitivity(Ste7_num_index ).value * 10) '")'];
    model.fname = ['10xSte7'];             
    write_and_execute_bngl_file(model, expt, obs, new_params);

    
    %Compile BNGL commands to set the estimated parameters 
    new_params={};
    for j = 1:length(params_for_sensitivity)
        new_params{end+1} = ['setParameter("' char(params_for_sensitivity(j).name) '","' ...
                 num2str(params_for_sensitivity(j).value) '")'];
    end
    %add the BNGL command to increase the Ste5_num by 10x
    new_params{end+1} = ['setParameter("Ste5_num","' ...
                     num2str(params_for_sensitivity(Ste5_num_index ).value * 10) '")'];
    model.fname = ['10xSte5'];             
    write_and_execute_bngl_file(model, expt, obs, new_params);

    %add the BNGL command to increase the Ste7_num by 10x
    new_params{end+1} = ['setParameter("Ste7_num","' ...
                     num2str(params_for_sensitivity(Ste7_num_index ).value * 10) '")'];
    model.fname = ['10xSte5_10xSte7'];             
    write_and_execute_bngl_file(model, expt, obs, new_params);

    
    
    
    model.fname = ['WT'];
    [sim_WT expt] = read_sim_results(model, expt, obs);        
    model.fname = ['10xSte7'];             
    [sim_Ste7 expt] = read_sim_results(model, expt, obs);        
    model.fname = ['10xSte5'];
    [sim_Ste5 expt] = read_sim_results(model, expt, obs);        
    model.fname = ['10xSte5_10xSte7'];             
    [sim_Ste5Ste7 expt] = read_sim_results(model, expt, obs);        

    %get the induced and basal pathway output
    for expt_num = 1:length(sim_Ste5)
        if isequal(expt{expt_num}{2}.name, '100 nM pheromone')
            data_index = expt{expt_num}{2}.obs{1}.data_index;
            sig_WT = sim_WT{expt_num}{2}.Data(end,data_index)/num2uM
            sig_Ste7 = sim_Ste7{expt_num}{2}.Data(end,data_index)/num2uM
            sig_Ste5 = sim_Ste5{expt_num}{2}.Data(end,data_index)/num2uM
            sig_Ste5Ste7 = sim_Ste5Ste7{expt_num}{2}.Data(end,data_index)/num2uM
        end
    end
    
    sig_Ste7 / sig_WT
    sig_Ste5Ste7 / sig_Ste5
    
end








function write_and_execute_bngl_file(model, expt, obs, new_params)

    %Simulate each experiment in expt
    for expt_num = 1:length(expt)

        %Start constructing the contents of a BNGL file to run the simulations
        %for the current experiment.  The first line tells BNG to load the
        %model network file (which was generated in Main_estimate_params.m.
        bngl_contents = {['readFile({file=>"' model.BNGL_fname '.net"})']};

        %Add to the conents of the new BNGL file the BNGL statements that we 
        %previously constructed to assign protein abundances and parameter
        %values.
        for i = 1:length(new_params)
            bngl_contents{end+1} = new_params{i};
        end

        %For each part of the current experiment, add the part-specific
        %commands to the BNGL file.
        for expt_part = 1:length(expt{expt_num})

                %set protein concentrations and parameters specific to the 
                %current experiment part (for example, setting dose information
                %via G protein activation/deactivation rates).
            for i = 1:length(expt{expt_num}{expt_part}.species.names)
                bngl_contents{end+1} = ['setConcentration("' char(expt{expt_num}{expt_part}.species.names(i)) '","' ...
                                char(expt{expt_num}{expt_part}.species.values{i}) '")'];
            end

            for i = 1:length(expt{expt_num}{expt_part}.param.names)
                bngl_contents{end+1} = ['setParameter("' char(expt{expt_num}{expt_part}.param.names(i)) '","' ...
                                char(expt{expt_num}{expt_part}.param.values{i}) '")'];
            end

            stoptime = expt{expt_num}{expt_part}.duration;

            %add the command to simulate the model for the current part of the
            %experiment
            bngl_contents{end+1} = ['simulate_ode({prefix=>"expt_' num2str(expt_num) '_part_' num2str(expt_part) ' - ' model.fname '"' ...
                                    ',t_end=>' num2str(stoptime) ...
                                    ',atol=>' num2str(model.AbsoluteTolerance) ...
                                    ',rtol=>' num2str(model.RelativeTolerance) ...
                                    ',n_steps=>' num2str(expt{expt_num}{expt_part}.num_t) ...
                                    ',verbose=>0});'];
            bngl_contents{end+1} = '';


        end

        %open a file to write the current experiment BNGL file.
        fid = fopen(['expt_' num2str(expt_num) '_sim - ' model.fname '.bngl'], 'w');
        for line_num = 1:length(bngl_contents)
            fprintf(fid, '%s\n', char(bngl_contents{line_num}));
        end
        fclose(fid);
        
        [s, w] = system(['expt_' num2str(expt_num) '_sim - ' model.fname '.bngl']);

    end

end



function [sim expt] = read_sim_results(model, expt, obs)

    %Simulate each experiment in expt
    for expt_num = 1:length(expt)

        %Here we open the results file generated by the simulations.  There is
        %one simulation results file per part of the current experiment.
        for expt_part = 1:length(expt{expt_num})

            %Construct the results file name and read in the contents of the
            %file.
            data_file_name = ['expt_' num2str(expt_num) '_part_' num2str(expt_part) ' - ' model.fname '.gdat']
            [tempData,tempHeaders] = tblread([data_file_name]);

            %Store the headings of the columns of the results file in the sim
            %object.
            for i = 1:size(tempHeaders,1)-1
                sim{expt_num}{expt_part}.headers{i} = strtrim(tempHeaders(i+1,:));
            end

            %Store the observables from the simulation in the sim object.
            sim{expt_num}{expt_part}.Data = tempData(:,2:end);

            %Here we figure out which observable in the expt object
            %(expt{}{}.obs{}) corresponds with which column of the simulation
            %results (sim{}{}.ts).  The corresponding column number of 
            %sim{}{}.ts is stored in expt{}{}.obs{}.data_index().
            for expt_obs_num = 1:length(expt{expt_num}{expt_part}.obs)
                expt{expt_num}{expt_part}.obs{expt_obs_num}.data_index = [];

                for header_num = 1:length(sim{expt_num}{expt_part}.headers)

                    obs_index = expt{expt_num}{expt_part}.obs{expt_obs_num}.obs_index;
                    for k = 1:length(obs{obs_index}.observables)
                        if isequal(obs{obs_index}.observables{k},sim{expt_num}{expt_part}.headers{header_num})
                            expt{expt_num}{expt_part}.obs{expt_obs_num}.data_index(end+1) = header_num;
                        end
                    end
                end
            end

        end

    end
end