function [F scale shift] = fit_data_to_sim(sim, expt, obs, new_params)

%Get the fitness score (F), scale and shift values for each observable
%based on a best fit of the data by scaling, and shifting if neccessary,
%the data to match the current simulation results in 'sim'.
F = zeros(size(obs));
scale = zeros(size(obs));
shift = zeros(size(obs));


for obs_num = 1:length(obs)
    if ~isfield(obs{obs_num}, 'fit_type')
        %do nothing, field does not exist
    elseif isequal(obs{obs_num}.fit_type, 'scale and shift')
        [F(obs_num) scale(obs_num) shift(obs_num)] = scale_and_shift(sim, expt, obs, obs_num, new_params);
    elseif isequal(obs{obs_num}.fit_type, 'scale no shift')
        [F(obs_num) scale(obs_num)] = scale_no_shift(sim, expt, obs, obs_num, new_params);
        shift(obs_num)=0;
    else
        msgbox(obs{obs_num}.fit_type)
        ERROR
    end
end





%**********************************************************************************    
function [F scale shift] = scale_and_shift(sim, expt, obs, obs_num, new_params)

% In this function, use lsqlin to minimize ||Cx-d||^2.  
% x = [scale; shift]
% C = [sim_data_column column_of_ones]
% d = [data_column]
% This scales abnd shifts the data to maximize the agreement between the
% experimental results and simulation results.

    %get the initial guesses for the scale and shift values
    x0 = [eval(obs{obs_num}.data_scale.value); eval(obs{obs_num}.data_shift.value)];

    %no inequality constraint A * x <= beq
    A = [];
    b = [];

    %no equality constraint Aeq * x = beq
    Aeq=[];
    beq=[];

   
    %set bounds on x
    lb = [eval(obs{obs_num}.data_scale.lb); eval(obs{obs_num}.data_shift.lb)];
    ub = [eval(obs{obs_num}.data_scale.ub); eval(obs{obs_num}.data_shift.ub)];


    %here we go through all experiments, and compile all measured data and
    %corresponding simulation results into column vectors.
    data_vect=[];
    sim_vect=[];
    for expt_num = 1:length(expt)
        for expt_part = 1:length(expt{expt_num})
            for expt_obs_num = 1:length(expt{expt_num}{expt_part}.obs)
                
                    %If the current observable has corresponding data, and
                    %this observable matches with the observable we're
                    %considering in this function call, then put sim
                    %results and data into the vectors.
                if isfield(expt{expt_num}{expt_part}.obs{expt_obs_num},'data') && ...
                        isequal(obs{obs_num}.name, expt{expt_num}{expt_part}.obs{expt_obs_num}.name)
                    data_vect = [data_vect; expt{expt_num}{expt_part}.obs{expt_obs_num}.data(:)];
                    new_ts = resample(sim{expt_num}{expt_part}.ts, expt{expt_num}{expt_part}.obs{expt_obs_num}.t_data) ;
                    sim_vect = [sim_vect; sum(new_ts.Data(:,expt{expt_num}{expt_part}.obs{expt_obs_num}.data_index),2)];
                end
            end
        end
    end
    
    C = [data_vect ones(size(data_vect))];
    d = sim_vect;

    opt = optimset('LargeScale', 'off', 'Display', 'off');

    %optimize the values of x = [scale; shift] to maximize agreement
    %between simulation and experiments
    [x resnorm] = lsqlin(C,d,A,b,Aeq,beq,lb,ub,x0,opt);

    scale = x(1);

    shift = x(2);

    %So that the value of the objective function (F) is not influenced by
    %the concentration (ie lower concentration leads to lower F), we will
    %calculate F by scaling the simulation results to match the data (which is
    %unchanging).
    %For lsqlin, resnorm = norm(data_vect * scale + shift - sim_vect)^2
    %We want units inside of norm to be the units of data_vect.
    %Thus, F = norm(data_vect - (sim_vect-shift)/scale)^2

    tmp = data_vect - (sim_vect-shift)/scale;

    F = sqrt(sum(tmp.^2)/length(tmp));
    
    
    
    
    
%**********************************************************************************    
function [F scale] = scale_no_shift(sim, expt, obs, obs_num, new_params)
% In this function, use lsqlin to minimize ||Cx-d||^2.  
% x = [scale]
% C = [sim_data_column]
% d = [data_column]
% This scales abnd shifts the data to maximize the agreement between the
% experimental results and simulation results.

    %get the initial guesses for the scale value
    x0 = [eval(obs{obs_num}.data_scale.value)];

    %no inequality constraint A * x <= beq
    A = [];
    b = [];

    %no equality constraint Aeq * x = beq
    Aeq=[];
    beq=[];

    %set bounds on x
    lb = eval(obs{obs_num}.data_scale.lb);
    ub = eval(obs{obs_num}.data_scale.ub);


    %here we go through all experiments, and compile all measured data and
    %corresponding simulation results into column vectors.
    data_vect=[];
    sim_vect=[];
    for expt_num = 1:length(expt)
        for expt_part = 1:length(expt{expt_num})
            for expt_obs_num = 1:length(expt{expt_num}{expt_part}.obs)

                    %If the current observable has corresponding data, and
                    %this observable matches with the observable we're
                    %considering in this function call, then put sim
                    %results and data into the vectors.
                if isfield(expt{expt_num}{expt_part}.obs{expt_obs_num},'data') && ...
                        isequal(obs{obs_num}.name, expt{expt_num}{expt_part}.obs{expt_obs_num}.name)
                    data_vect = [data_vect; expt{expt_num}{expt_part}.obs{expt_obs_num}.data(:)];
                    new_ts = resample(sim{expt_num}{expt_part}.ts, expt{expt_num}{expt_part}.obs{expt_obs_num}.t_data) ;
                    sim_vect = [sim_vect; sum(new_ts.Data(:,expt{expt_num}{expt_part}.obs{expt_obs_num}.data_index),2)];
                end
            end
        end
    end

    C = [data_vect];
    d = sim_vect;

    opt = optimset('LargeScale', 'off', 'Display', 'off');

    %optimize the values of x = [scale; shift] to maximize agreement
    %between simulation and experiments
    [x resnorm] = lsqlin(C,d,A,b,Aeq,beq,lb,ub,x0,opt);

    scale = x;

    %So that the value of the objective function (F) is not influenced by
    %the concentration (ie lower concentration leads to lower F), we will
    %calculate F by scaling the simulation results to match the data (which is
    %unchanging).
    %For lsqlin, resnorm = norm(data_vect * scale + shift - sim_vect)^2
    %We want units inside of norm to be the units of data_vect.
    %Thus, F = norm(data_vect - (sim_vect-shift)/scale)^2

    tmp = data_vect - (sim_vect)/scale;

    F = sqrt(sum(tmp.^2)/length(tmp));

    
       
