close all
clear all
drawnow

tic

%The cell volume is 29fL (as in the model)
Cell_volume=29e-15;
Avogadros_number=6.022e23;

%a factor to convert #/cell to uM (and vice versa)
num2uM = 1e6/(Cell_volume * Avogadros_number);

%Here we list all the results files for the simulations that we ran.
fnames{1} = { ...
   'YMC - 1 - 07-Feb-2008.mat', ...
   'YMC - 2 - 07-Feb-2008.mat', ...
   'YMC - 3 - 07-Feb-2008.mat', ...
   'YMC - 4 - 07-Feb-2008.mat', ...
       };
save_name_root{1} = 'YMC';    

fnames{2} = { ...
   'YMC_Gmgm - 1 - 07-Feb-2008.mat', ...
   'YMC_Gmgm - 2 - 07-Feb-2008.mat', ...
       };
save_name_root{2} = 'YMC_Gmgm';    

fnames{3} = { ...
   'YMC_Sltr - 1 - 07-Feb-2008.mat', ...
   'YMC_Sltr - 2 - 07-Feb-2008.mat', ...
       };
save_name_root{3} = 'YMC_Sltr';    

fnames{4} = { ...
   'YMC_Mdr - 1 - 07-Feb-2008.mat', ...
   'YMC_Mdr - 2 - 07-Feb-2008.mat', ...
       };
save_name_root{4} = 'YMC_Mdr';    

fnames{5} = { ...
   'YMC_opt - 1 - 07-Feb-2008.mat', ...
   'YMC_opt - 2 - 07-Feb-2008.mat', ...
       };
save_name_root{5} = 'YMC_opt';    




%% Load GA results
%Here we load the simulation results, and combine all parameter sets and 
%from all generations during parameter estimation for a given protein 
%abundance set into a single array (same for scores).  because these arrays
%can be quite large, depending on the number of generations and the
%population size, we use singles instead of doubles.

%for each file set (aka, each parameter abundance set)
for file_set = 1:length(fnames)

    %Initialize variables to hold all scores and parameter values
    all_scores{file_set}=single([]);
    all_params{file_set}=single([]);
    
    %for each time the optimization was run
    for file_num = 1:length(fnames{file_set})

        %load the optimization results
        load(fnames{file_set}{file_num});

        %reshape the parameters to put one generation after the next (ie,
        %get rid of the 'generations dimension' of the parameters variable.
        params_reshaped = reshape(params,size(params,1)*size(params,2),size(params,3));
        
        %append scores and parameters from the current optimization to the
        %scores and parameters from previous generations
        all_scores{file_set} = [all_scores{file_set}; score(:)];
        all_params{file_set} = [all_params{file_set}; params_reshaped];

    end

    %set up expt_obj for each abundance set
    expt_obj{file_set}.expt = expt;
    expt_obj{file_set}.param = param;
    expt_obj{file_set}.model = model;
    expt_obj{file_set}.obs = obs;
    expt_obj{file_set}.params_to_tune = params_to_tune;

    
end

%convert the bngl file into a network file.
[s, w] = system([expt_obj{1}.model.BNG2_path ' ' model.BNGL_fname '.bngl']);




%% Run simulations for Figure 6 and 7
%Vary the abundances of Ste11, Ste7, Fus3, and Ste5, and run the
%simulations for each measured protein abundance set.

%Sample the range of protein abundances more less densly near the max and
%min abundances considered.
log_spacing_10 = logspace(0,1,11);
log_spacing_20 = logspace(0,1,21);
log_spacing_30 = logspace(0,1,31);

Ste5_vals = [ ...
    4.8 * log_spacing_10(1:end-1), ...
    48 * log_spacing_20(1:end-1), ...
    480 * log_spacing_30(1:end-1), ...
    4800 * log_spacing_30(1:end-1), ...
    48000 * log_spacing_20(1:end-1), ...
    480000 * log_spacing_10(1:end-1), ...
    4800000 1e7];

Ste11_vals = [ ...
    3.2 * log_spacing_10(1:end-1), ...
    32 * log_spacing_20(1:end-1), ...
    320 * log_spacing_30(1:end-1), ...
    3200 * log_spacing_30(1:end-1), ...
    32000 * log_spacing_20(1:end-1), ...
    320000 * log_spacing_10(1:end-1), ...
    3200000, 1e7];

Ste7_vals = [ ...
    9.2 * log_spacing_10(1:end-1), ...
    92 * log_spacing_20(1:end-1), ...
    920 * log_spacing_30(1:end-1), ...
    9200 * log_spacing_30(1:end-1), ...
    92000 * log_spacing_20(1:end-1), ...
    920000 * log_spacing_10(1:end-1), ...
    1e7];

Fus3_vals = [ ...
    2 * log_spacing_10(1:end-1), ...
    20 * log_spacing_20(1:end-1), ...
    200 * log_spacing_30(1:end-1), ...
    2000 * log_spacing_30(1:end-1), ...
    20000 * log_spacing_30(1:end-1), ...
    200000 * log_spacing_20(1:end-1), ...
    2000000 * log_spacing_10(1:end-1)];
Fus3_vals = [Fus3_vals(find(Fus3_vals<1e7)) 1e7];

%Store the names of the abundances, parameter axes, and the protein
%abundance ranges.
abund_to_vary = {'Ste11_num', 'Ste7_num', 'Fus3_num', 'Ste5_num'};
abund_names_for_axes = {'# Ste11 per cell', '# Ste7 per cell', '# Fus3 per cell', '# Ste5 per cell'};
abund_ranges = {Ste11_vals, Ste7_vals, Fus3_vals, Ste5_vals};

%For each measured abundance set
for file_set = 1:length(fnames) 

    curr_expt_obj = expt_obj{file_set};

    %get the best fit parameter set
    [min_score min_index] = min(all_scores{file_set});
    params=all_params{file_set}(min_index,:);

    %for each protein abundance to vary
    for varying_abund_num = 1:length(abund_to_vary)

        display([save_name_root{file_set} ' - varying ' abund_to_vary{varying_abund_num} ' - elapsed time: ' num2str(toc) 's'])

        %run the simulations for the range of protein abundances
        [WT_amount basal_sig sig] = vary_abund(curr_expt_obj, abund_to_vary{varying_abund_num}, abund_ranges{varying_abund_num}, params);

        WT_amounts{file_set}(varying_abund_num) = WT_amount;

        %calculate and store the pathway output, induction ratio and swing
        basal_signal{file_set,varying_abund_num} = basal_sig/num2uM;
        signal{file_set,varying_abund_num} = [basal_sig/num2uM; sig/num2uM];
        IR{file_set,varying_abund_num} = sig ./ basal_sig;
        swing{file_set,varying_abund_num} = sig/num2uM - basal_sig/num2uM;

    end

end
%save the results
save Dynamic_range.mat basal_signal signal IR swing abund_to_vary abund_names_for_axes abund_ranges WT_amounts save_name_root




%% plot Figure 6
% Dynamic range vs Ste5 abundance for (a) Our, (b) Ghaemmaghami, (c)
% Slaughter and (d) Maeder abundance sets

load Dynamic_range.mat

signal_range = {[0 15000], [0 5000], [0 1000], [0 3000]};
swing_range = {[0 10000], [0 3000], [0 1000], [0 2000]};
IR_range = [1 100];


fig = figure('Position',[1 65 1280 909]);

for file_set=1:4

    
    protein_info.WT_amount = WT_amounts{file_set}(4);
    protein_info.abund_name=abund_to_vary{4};
    protein_info.abund_range=abund_ranges{4};
    protein_info.x_axis_label=abund_names_for_axes{4};

    data_to_plot.signal=signal{file_set,4};
    data_to_plot.swing=swing{file_set,4};
    data_to_plot.IR=IR{file_set,4};
    data_to_plot.signal_range=signal_range{file_set};
    data_to_plot.swing_range=swing_range{file_set};
    data_to_plot.IR_range=IR_range;

    plot_dynamic_range(fig, 4, file_set, protein_info, data_to_plot)
    
end    

saveas(fig, 'Fig6-dynamic_range_vs_Ste5.fig')
close(fig)
drawnow;




%% plot Figure 7 and Figure S5a
% Figure 7 is a stripped down version of S5, with swing plots removed in
% Illustrator.
% Dynamic range vs abundance of (a) Ste11, (b) Ste7, and (c) Fus3 using our
% measured abundances

load Dynamic_range.mat

signal_range = {[0 10000], [0 10000], [0 15000]};
swing_range = {[0 10000], [0 10000], [0 15000]};
IR_range = [1 100];

fig = figure('Position',[1 65 1280 909]);

file_set=1;
for protein_num=1:3

    protein_info.WT_amount = WT_amounts{file_set}(protein_num);
    protein_info.abund_name=abund_to_vary{protein_num};
    protein_info.abund_range=abund_ranges{protein_num};
    protein_info.x_axis_label=abund_names_for_axes{protein_num};

    data_to_plot.signal=signal{file_set,protein_num};
    data_to_plot.swing=swing{file_set,protein_num};
    data_to_plot.IR=IR{file_set,protein_num};
    data_to_plot.signal_range=signal_range{protein_num};
    data_to_plot.swing_range=swing_range{protein_num};
    data_to_plot.IR_range=IR_range;

    plot_dynamic_range(fig, 4, protein_num, protein_info, data_to_plot)
    
end    

saveas(fig, 'Fig7-dynamic_range_vs_kinases.fig')
close(fig)
drawnow;




%% Fig S4, fit of models to data
%Plot fit with data for Ste5 translocation (left column) and Fus3 
%phosphorylation (right column) for the best fit parameter set for each 
%protein abundance set (rows)

fig = figure('Position',[200 100 700 800]);
for file_set = 1:length(fnames) 
    curr_expt_obj = expt_obj{file_set};

    %identify the best-fit parameter set (lowest score)
    [min_score min_index] = min(all_scores{file_set});
    params = all_params{file_set}(min_index,:);

    %simulate and plot the results
    plot_dose_resp(fig, curr_expt_obj, params, length(fnames), file_set, save_name_root{file_set}, num2uM)
    
end
  
saveas(fig, 'FigS4-fits.fig')
close(fig)
drawnow;



%% Supplemental Figs - dynamic range for different param sets for Benjamin abundances
%Here we identify alternate parameter sets that give a good fit to the data
%and are as different as possible from the overall best-fit parameter set.
%For our abundance set only!

%Scores of parameters sets to be considered must be within 10% (1.1x) the
%score of the best fit.  These are called good-fit parameters or top
%parameters.
top_params_threshhold = 1.1;

%We consider 4 additional parameter sets, so 5 sets in total including
%best-fit.
num_clusters=5;

%due to computational limitations, so can only consider ~5000 good
%parameter sets.
max_num_sets = 5000;

%get the best-fit parameter set
[min_score min_index] = min(all_scores{1});

%identify all good-fit parameter sets
top_sets = find(all_scores{1}<=min_score*top_params_threshhold);

%Find out which of good-fit parameter sets is the best-fit.
min_index = find(top_sets==min_index);

%randomize the good-fit parameter sets so we select random top 5000 sets.
[y, I] = sort(rand(size(top_sets(:))));

%if there are more than 5000 good parameter sets
if length(I)>=max_num_sets
    %if the best-fit set is not part of the first 5000 good-fit parameter
    %sets, add it to the list.
    if isempty(find(I(1:max_num_sets)==min_index,1))
        I(1)=min_index;
    end

    %filtered list is only 5000 of the good-fit parameter sets
    top_sets_filtered = top_sets(I(1:max_num_sets)); 
else
    top_sets_filtered = top_sets(I);        
end
top_set = top_sets(I);        

%store the good-fit parameters and scores
top_scores_filtered = all_scores{1}(top_sets_filtered);
top_params_filtered = all_params{1}(top_sets_filtered,:);


%Now we use a linkage map to identify the different clusters/subsets of the
%parameter sets, from which we'll select the alternate parameter sets to
%use.  We use the log of the parameter values to give equal weighting to
%all parameters.
log_top_params = log10(top_params_filtered);
distances = pdist(log_top_params);
links = linkage(distances);
T = cluster(links,'maxclust',num_clusters);

%pull out the members of each cluster and find the best parameter set from 
%each cluster
for i=1:num_clusters
    cluster_members{i} = find(T==i);
    min_set_score(i) = min(top_scores_filtered(cluster_members{i}));
end
[y sorted_cluster_indexes] = sort(min_set_score);

%store the best parameter set from each cluster in param_sets_to_examine
for i=1:num_clusters
    temp_cluster{i}=cluster_members{sorted_cluster_indexes(i)};
end
cluster_members=temp_cluster;
for i=1:num_clusters
    display(['Cluster ' num2str(i) ', ' num2str(length(cluster_members{i})) ' parameter sets, ' num2str(min(top_scores_filtered(cluster_members{i}))) ' minimum score, ' num2str(std(top_scores_filtered(cluster_members{i}))) ' standard deviation']);
    [min_val I] = min(top_scores_filtered(cluster_members{i}));
    param_sets_to_examine(i) = cluster_members{i}(I);
end
    
clear scores_by_run params_by_run distances




%% Rest of Figure S4 - plots of fits using different parameter sets (Benjamin abundances)
%Plot fit with data for Ste5 translocation (left column) and Fus3 
%phosphorylation (right column) for the best fit parameter set for each 
%protein abundance set (rows)

fig = figure('Position',[200 100 700 800]);
curr_expt_obj = expt_obj{1};
for cluster_num = 2:5 

    params = top_params_filtered(param_sets_to_examine(cluster_num),:);
    
    %simulate and plot the results
    plot_name = ['Parameter set ' num2str(cluster_num)];
    plot_dose_resp(fig, curr_expt_obj, params, 5, cluster_num-1, plot_name, num2uM)
    
end

saveas(fig, ['FigS4-Benjamin_cluster_fits.fig'])
close(fig)    
drawnow;





%% Run simulations for figure S5b-e
%Vary the abundances of Ste11, Ste7, Fus3, and Ste5, and run the
%simulations for each alternate parameter set for our measured protein 
%abundances.

%load the results of varying the protein abundaces previously (to get
%ranges of protein abundances, etc).
load Dynamic_range.mat

%Grab the previous results for varying protein abundances for the best-fit
%parameter set.
file_set=1;
for varying_abund_num=1:4
    basal_signal_clust{1,varying_abund_num} = basal_signal{file_set,varying_abund_num};
    signal_clust{1,varying_abund_num} = signal{file_set,varying_abund_num};
    IR_clust{1,varying_abund_num} = IR{file_set,varying_abund_num};
    swing_clust{1,varying_abund_num} = swing{file_set,varying_abund_num};
end

clear params WT_amounts
params{1} = top_params_filtered(param_sets_to_examine(1),:);
curr_expt_obj = expt_obj{1};

%For each alternate parameter set
for cluster_num=2:num_clusters

    %store the parameter values to use
    params{cluster_num} = top_params_filtered(param_sets_to_examine(cluster_num),:);

    for varying_abund_num = 1:length(abund_to_vary)

        display(['Benjamin abundances - cluster ' num2str(cluster_num) ' - varying ' abund_to_vary{varying_abund_num} ' - elapsed time: ' num2str(toc) 's'])
        
        %run the simulations for the range of protein abundances
        [WT_amount basal_sig sig] = vary_abund(curr_expt_obj, abund_to_vary{varying_abund_num}, abund_ranges{varying_abund_num}, params{cluster_num});

        WT_amounts(varying_abund_num) = WT_amount;

        %calculate and store the pathway output, induction ratio and swing
        basal_signal_clust{cluster_num,varying_abund_num} = basal_sig/num2uM;
        signal_clust{cluster_num,varying_abund_num} = [basal_sig/num2uM; sig/num2uM];
        IR_clust{cluster_num,varying_abund_num} = sig ./ basal_sig;
        swing_clust{cluster_num,varying_abund_num} = sig/num2uM - basal_sig/num2uM;

    end

end

%save the simulation results
save(['Benjamin_clusters_data.mat'], 'basal_signal_clust', 'signal_clust', 'IR_clust', 'swing_clust', 'abund_to_vary', 'abund_names_for_axes', 'abund_ranges', 'WT_amounts', 'params');
clear params



%% plot Figure S5 
% Dynamic range vs abundance of Ste11, Ste7, Fus3 and Ste5
% using our measured abundances.  Each plot is for a different parameter
% set.

%load simulation results
load Benjamin_clusters_data.mat

signal_range = [0 20000];
swing_range = [0 20000];
IR_range = [1 100];

file_set=1;

for cluster_num=2:5
    fig = figure('Position',[1 65 1280 909]);

    for protein_num=1:4

        protein_info.WT_amount = WT_amounts(protein_num);
        protein_info.abund_name=abund_to_vary{protein_num};
        protein_info.abund_range=abund_ranges{protein_num};
        protein_info.x_axis_label=abund_names_for_axes{protein_num};

        data_to_plot.signal=signal_clust{cluster_num,protein_num};
        data_to_plot.swing=swing_clust{cluster_num,protein_num};
        data_to_plot.IR=IR_clust{cluster_num,protein_num};
        data_to_plot.signal_range=signal_range;
        data_to_plot.swing_range=swing_range;
        data_to_plot.IR_range=IR_range;

        plot_dynamic_range(fig, 4, protein_num, protein_info, data_to_plot)
    end
    plot_name = ['FigS5-Benjamin_cluster_' num2str(cluster_num) '_dynamic_range.fig'];
    saveas(fig, plot_name)
    close(fig)
    drawnow;

end



%% plot Figure S6a-d 
% Dynamic range vs abundance of Ste11, Ste7, Fus3 and Ste5
% using the abundances measure by (S6a) Ghaemmaghami et al., (S6b)
% Slaughter et al., (S6c) Maeder et al., and (S6d) the abundances optimized
% during parameter optimization.

load Dynamic_range.mat

signal_range = {[0    0], [0    0], [0     0], [0    0]; ...
                [0 5000], [0 5000], [0 20000], [0 5000]; ...
                [0 1000], [0 1000], [0  1500], [0 1000]; ...
                [0 2000], [0 2000], [0  2000], [0 3000]; ...
                [0 1000], [0 1000], [0  5000], [0 2000]};
swing_range = signal_range;
IR_range = [1 100];


for file_set=2:5

    fig = figure('Position',[1 65 1280 909]);

    for protein_num=1:4

        protein_info.WT_amount = WT_amounts{file_set}(protein_num);
        protein_info.abund_name=abund_to_vary{protein_num};
        protein_info.abund_range=abund_ranges{protein_num};
        protein_info.x_axis_label=abund_names_for_axes{protein_num};

        data_to_plot.signal=signal{file_set,protein_num};
        data_to_plot.swing=swing{file_set,protein_num};
        data_to_plot.IR=IR{file_set,protein_num};
        data_to_plot.signal_range=signal_range{file_set,protein_num};
        data_to_plot.swing_range=swing_range{file_set,protein_num};
        data_to_plot.IR_range=IR_range;

        plot_dynamic_range(fig, 4, protein_num, protein_info, data_to_plot)
    end
    plot_name = ['FigS6-' save_name_root{file_set} '_dynamic_range.fig'];

    saveas(fig, plot_name)
    close(fig)
    drawnow;

end




%perform sensitivity analysis

curr_expt_obj = expt_obj{1};

params_for_sensitivity(1).name = 'Fus3_num';
params_for_sensitivity(1).value = 20400;
params_for_sensitivity(end+1).name = 'kbind_Gpa1_Ste4_0_1nM';
params_for_sensitivity(end).value = 0.37091;
params_for_sensitivity(end+1).name = 'kbind_Gpa1_Ste4_0nM';
params_for_sensitivity(end).value = 0.10273;
params_for_sensitivity(end+1).name = 'kbind_Gpa1_Ste4_100nM';
params_for_sensitivity(end).value = 0.014296;
params_for_sensitivity(end+1).name = 'kcat_Msg5_MAPK_PO4';
params_for_sensitivity(end).value = 0.062661;
params_for_sensitivity(end+1).name = 'kcat_nonspecific_dephosph';
params_for_sensitivity(end).value = 0.0021919;
params_for_sensitivity(end+1).name = 'kcat_Ptp_MAPK_PO4';
params_for_sensitivity(end).value = 2.8774;
params_for_sensitivity(end+1).name = 'kcat_Ste11pSpSpTSte5Ste7_pS';
params_for_sensitivity(end).value = 3.6382;
params_for_sensitivity(end+1).name = 'kcat_Ste11pSpSpTSte5Ste7pS_pT';
params_for_sensitivity(end).value = 4.5313;
params_for_sensitivity(end+1).name = 'kcat_Ste4Ste18Ste5Ste11_pS';
params_for_sensitivity(end).value = 2.7582;
params_for_sensitivity(end+1).name = 'kcat_Ste5Ste7pSpTFus3_pT';
params_for_sensitivity(end).value = 0.70423;
params_for_sensitivity(end+1).name = 'kcat_Ste5Ste7pSpTFus3_pY';
params_for_sensitivity(end).value = 4.4084;
params_for_sensitivity(end+1).name = 'kcat_Ste5Ste7pSpTFus3pT_pY';
params_for_sensitivity(end).value = 0.48499;
params_for_sensitivity(end+1).name = 'kcat_Ste5Ste7pSpTFus3pY_pT';
params_for_sensitivity(end).value = 5.051;
params_for_sensitivity(end+1).name = 'Kd_Msg5_MAPK';
params_for_sensitivity(end).value = 1.4;
params_for_sensitivity(end+1).name = 'Kd_Ste5_Fus3';
params_for_sensitivity(end).value = 1;
params_for_sensitivity(end+1).name = 'Kd_Ste5_Ste11';
params_for_sensitivity(end).value = 0.140124976;
params_for_sensitivity(end+1).name = 'Kd_Ste5_Ste7';
params_for_sensitivity(end).value = 0.001952477;
params_for_sensitivity(end+1).name = 'Kd_Ste7_MAPK';
params_for_sensitivity(end).value = 0.1;
params_for_sensitivity(end+1).name = 'kdilution';
params_for_sensitivity(end).value = 0.000128;
params_for_sensitivity(end+1).name = 'koff_Gpa1_Ste4_0_1nM';
params_for_sensitivity(end).value = 0.034849;
params_for_sensitivity(end+1).name = 'koff_Gpa1_Ste4_0nM';
params_for_sensitivity(end).value = 0.00039613;
params_for_sensitivity(end+1).name = 'koff_Gpa1_Ste4_100nM';
params_for_sensitivity(end).value = 0.048251;
params_for_sensitivity(end+1).name = 'koff_Ptp_MAPK';
params_for_sensitivity(end).value = 0.064643;
params_for_sensitivity(end+1).name = 'koff_Ste4Ste18_Ste5';
params_for_sensitivity(end).value = 0.26373;
params_for_sensitivity(end+1).name = 'koff_Ste5_Fus3';
params_for_sensitivity(end).value = 2.3;
params_for_sensitivity(end+1).name = 'kon_Msg5_MAPK';
params_for_sensitivity(end).value = 0.92293;
params_for_sensitivity(end+1).name = 'kon_Ptp_MAPK';
params_for_sensitivity(end).value = 1.0082;
params_for_sensitivity(end+1).name = 'kon_Ste4Ste18_Ste5';
params_for_sensitivity(end).value = 1.8799;
params_for_sensitivity(end+1).name = 'kon_Ste5_Ste11';
params_for_sensitivity(end).value = 0.10562;
params_for_sensitivity(end+1).name = 'kon_Ste5_Ste7';
params_for_sensitivity(end).value = 3.775;
params_for_sensitivity(end+1).name = 'kon_Ste7_MAPK';
params_for_sensitivity(end).value = 3.5466;
params_for_sensitivity(end+1).name = 'Msg5_num';
params_for_sensitivity(end).value = 38;
params_for_sensitivity(end+1).name = 'Ptp_num';
params_for_sensitivity(end).value = 193.6644;
params_for_sensitivity(end+1).name = 'Ste11_num';
params_for_sensitivity(end).value = 3500;
params_for_sensitivity(end+1).name = 'Ste11_pS_only_PO4_factor';
params_for_sensitivity(end).value = 2089.636;
params_for_sensitivity(end+1).name = 'Ste4_num';
params_for_sensitivity(end).value = 818;
params_for_sensitivity(end+1).name = 'Ste5_num';
params_for_sensitivity(end).value = 480;
params_for_sensitivity(end+1).name = 'Ste7_num';
params_for_sensitivity(end).value = 920;
params_for_sensitivity(end+1).name = 'Ste7_pS_only_PO4_factor';
params_for_sensitivity(end).value = 3871.1816;
params_for_sensitivity(end+1).name = 'Cell_volume';
params_for_sensitivity(end).value = 2.9E-14;

results{1,1} = 'parameter';
results{1,2} = 'delta(dynamic range)';
results{1,3} = 'delta(pathway output)';

for param_val_num = 1:length(params_for_sensitivity)

    new_params={};

    %Compile BNGL commands to set the estimated parameters 
    for j = 1:length(params_for_sensitivity)
        if j~= param_val_num
            new_params{end+1} = ['setParameter("' char(params_for_sensitivity(j).name) '","' ...
                     num2str(params_for_sensitivity(j).value) '")'];
        end
    end

    %add the BNGL command to set the parameter whose sensitivity we are measuring
    new_params{end+1} = ['setParameter("' params_for_sensitivity(param_val_num).name '","' ...
                     num2str(params_for_sensitivity(param_val_num).value * 0.95) '")'];

    [sim_low curr_expt_obj.expt] = sim_expts(curr_expt_obj.model, curr_expt_obj.expt, curr_expt_obj.obs, new_params);

    %add the BNGL command to set the parameter whose sensitivity we are measuring
    new_params{end+1} = ['setParameter("' params_for_sensitivity(param_val_num).name '","' ...
                     num2str(params_for_sensitivity(param_val_num).value * 1.05) '")'];
    
    [sim_high curr_expt_obj.expt] = sim_expts(curr_expt_obj.model, curr_expt_obj.expt, curr_expt_obj.obs, new_params);

    
        %get the induced and basal pathway output
    data_index = curr_expt_obj.expt{1}{2}.obs{1}.data_index;
    sig_low = sim_low{1}{2}.Data(end,data_index);
    sig_high = sim_high{1}{2}.Data(end,data_index);

    basal_sig_low = sim_low{1}{1}.Data(end,data_index);
    basal_sig_high = sim_high{1}{1}.Data(end,data_index);

    delta_dynamic_range(param_val_num) = (sig_high / basal_sig_high) - (sig_low / basal_sig_low);
    delta_pathway_output(param_val_num) = (sig_high - sig_low);

    results{param_val_num+1,1} = params_for_sensitivity(param_val_num).name;
    results{param_val_num+1,2} = delta_dynamic_range(param_val_num);
    results{param_val_num+1,3} = delta_pathway_output(param_val_num);
end



Ste5_num_index = 38;
Ste7_num_index = 39;
%Compile BNGL commands to set the estimated parameters 
new_params={};
for j = 1:length(params_for_sensitivity)
    new_params{end+1} = ['setParameter("' char(params_for_sensitivity(j).name) '","' ...
             num2str(params_for_sensitivity(j).value) '")'];
end
[sim_WT curr_expt_obj.expt] = sim_expts(curr_expt_obj.model, curr_expt_obj.expt, curr_expt_obj.obs, new_params);

%add the BNGL command to increase the Ste7_num by 10x
new_params{end+1} = ['setParameter("Ste7_num","' ...
                 num2str(params_for_sensitivity(Ste7_num_index ).value * 10) '")'];
[sim_10xSte7 curr_expt_obj.expt] = sim_expts(curr_expt_obj.model, curr_expt_obj.expt, curr_expt_obj.obs, new_params);


%Compile BNGL commands to set the estimated parameters 
new_params={};
for j = 1:length(params_for_sensitivity)
    new_params{end+1} = ['setParameter("' char(params_for_sensitivity(j).name) '","' ...
             num2str(params_for_sensitivity(j).value) '")'];
end
%add the BNGL command to increase the Ste5_num by 10x
new_params{end+1} = ['setParameter("Ste5_num","' ...
                 num2str(params_for_sensitivity(Ste5_num_index ).value * 10) '")'];
model.fname = ['10xSte5'];             
[sim_10xSte5 curr_expt_obj.expt] = sim_expts(curr_expt_obj.model, curr_expt_obj.expt, curr_expt_obj.obs, new_params);

%add the BNGL command to increase the Ste7_num by 10x
new_params{end+1} = ['setParameter("Ste7_num","' ...
                 num2str(params_for_sensitivity(Ste7_num_index ).value * 10) '")'];
[sim_10xSte5_10xSte7 curr_expt_obj.expt] = sim_expts(curr_expt_obj.model, curr_expt_obj.expt, curr_expt_obj.obs, new_params);

%get the induced and basal pathway output
data_index = expt{1}{2}.obs{1}.data_index;
sig_WT = sim_WT{1}{2}.Data(end,data_index)/num2uM
sig_Ste7 = sim_10xSte7{1}{2}.Data(end,data_index)/num2uM
sig_Ste5 = sim_10xSte5{1}{2}.Data(end,data_index)/num2uM
sig_Ste5Ste7 = sim_10xSte5_10xSte7{1}{2}.Data(end,data_index)/num2uM

display(['Ratio of output with 10xSte7 to WT output = ' num2str(sig_Ste7 / sig_WT)])
display(['Ratio of output with 10xSte7 and 10xSte7 to output with 10xSte5 = ' num2str(sig_Ste5Ste7 / sig_Ste5)])