%% Electrode Reorder Script
% This script takes the output of iELVis electrode placement in BioImage
% Suite 35, assigns coordinates to each electrode, reorders the electrodes
% + coordinates according to the patient's montage, and converts the
% coordinates from native FreeSurfer space to AFNI/SUMA space. Both FS and
% AFNI coordinates are saved as .csv files in the current directory.
%
% Written August 20th, 2019 by Elizabeth "Buffy" Nesbitt
% Conversion script afni2elvis.m written by Eleonora Bartoli in September 2018
%
%Requires:
%   -patient's montage in .xlsx, .csv, or .txt format
%   -subject's subID.PIAL/.LEPTO and subID.electrodeNames files output by
%     BioImage Suite (which in turn means that all previous steps of the
%     pipeline have been finished)
%% Read in montage, pial coordinates, and electrode labels

%Clean up first
clear all
close all
clc

%Set variables for individual subject
subID = 'YCJ'; 
%For beauchamplab server:
    subDir = ['/Volumes/data/UT/' subID '/'];
    montageDir = subDir;
    elocDir = [subDir 'iELVis_Localization/' subID '/elec_recon/'];
% %For ecog server:
% subDir = ['/Volumes/ecog/ECoG_Data/' subID 'Datafile/']; %for ecog server
%     montageDir = [subDir 'INFO/'];
%     elocDir = [subDir 'IMG/' subID '/elec_recon/'];

%Read in patient's montage
montage = readtable([subDir subID '_montage_blocks8-14.xlsx']);
%montage = readtable([subDir 'YCK_montage.csv']); 

%Read in patient's .PIAL coordinates
pial_info = importdata([elocDir subID '.PIAL']);
pial_coords = array2table(pial_info.data(:,[1:3]));

%Read in patient's .electrodeNames labels
elec_label_info = importdata([elocDir subID '.electrodeNames']);
elec_names = elec_label_info([3:end],1); %skip header info in rows 1 and 2
elec_labels = cell2table(elec_names);

%Create table of labeled coordinates
labeled_pial = elec_labels(:,1);
labeled_pial(:,2:4) = pial_coords(:,1:3);

%% Reorder electrodes according to montage

%Set up index and labels
subCoords(:,1) = montage(:,1);
subCoords(:,2) = montage(:,3);
%Preallocate subCoords X, Y, and Z columns
subCoords(:,3) = array2table(zeros(size(subCoords(:,1))));
subCoords(:,4) = array2table(zeros(size(subCoords(:,1))));
subCoords(:,5) = array2table(zeros(size(subCoords(:,1))));

%convert table to cell because apparently MATLAB hates its own formatting
labeled_pial_temp{1,1} = [];
for i_row =1:size(labeled_pial,1); 
    labeled_pial_temp{i_row} = labeled_pial{i_row,1}{1,1}(1:end-4); 
end

%Find rows where iELVis label matches montage label
for i_row = 1:size(subCoords)
    try
        foo = find(strcmp(subCoords{i_row,2},labeled_pial_temp));
        subCoords(i_row, 3:5) = labeled_pial(foo, 2:4);
    end
end

%% afni2elvis conversion

%setup AFNI vs iELVis tables
fs_coords = table2array(subCoords(:,3:5));
subCoords_afni = subCoords(:,:);
subCoords_fs = subCoords(:,:);

%Perform conversion and update tables
afni_coords = afni2elvis(fs_coords,subID,'reverse');

subCoords_afni(:,3) = array2table(afni_coords(:,1));
subCoords_afni(:,4) = array2table(afni_coords(:,2));
subCoords_afni(:,5) = array2table(afni_coords(:,3));

%Return mini and empty channel coordinates to zeros
for i_row = 1:size(subCoords_afni)
    foo = find(fs_coords(i_row,3)==0);
    if foo==1
        subCoords_afni(i_row,3) = array2table(0);
        subCoords_afni(i_row,4) = array2table(0);
        subCoords_afni(i_row,5) = array2table(0);
    end
end

%% Save reordered electrode coordinates

%Rename headers
subCoords_fs.Properties.VariableNames = {'Channel', 'Label', 'Coord_x', 'Coord_y', 'Coord_z'};
subCoords_afni.Properties.VariableNames = {'Channel', 'Label', 'Coord_x', 'Coord_y', 'Coord_z'};

%Save file as .csv
writetable(subCoords_fs,[subDir subID 'electrodes_fs.csv']);
writetable(subCoords_afni,[subDir subID 'electrodes_afni.csv']);

%% afni2elvis function

function [output_coords]=afni2elvis(input_coords,subject,varargin)

% from a xyz coordinate matrix created in suma, change the coords so that they can be
% plotted using freesurfer/iELVIS functions (and viceversa)

% inputs: 
%--------------------------------------------------------------------------
% 1) a matrix with 3 cols and n rows: [x y z] the coordinates in afni/suma
% 2) the subject code identifier 
% (tricky: this will read the MRI header from the freesurfer subject folder 
% to figure out the trasformation needed)
% optional
% 3) varagin: 'reverse' (for now only coded option) to get the reverse transform: from
% ielvis/freesurfer to afni/suma

% outputs
%--------------------------------------------------------------------------
% output_coords: the coords into ielvis space (or afni/suma if using 'reverse')

% usage: 
%--------------------------------------------------------------------------
%  afni_coords = [46.5730 6.4382 -38.0190; 0 0 0];
% [elvis_coords]=afni2elvis(afni_coords,'YBN'); 

% or 

% elvis_coords = [-47.6677    2.1171  -45.9192; 0    0   0];
% [r_afni_coords]=afni2elvis(elvis_coords,'YBN','reverse'); 

% notes:
%--------------------------------------------------------------------------
% CHANGE THE VALUE of the following variable:
% directory_location = 'eb_local'; (line ~84)
% to any coded option in the switch, options coded for now are: 
% 'ECoG_server','Beauchamplab_server','all_local','StaticPattern'
% read below (lines 63 onward) to decide how to change this depending on where your data is
%--------------------------------------------------------------------------
% more notes:
%--------------------------------------------------------------------------
% input_coords: notes 
% afni/suma coordinates are in LPS format (xyz axis have higher numbers going left,posterior,superior)
% ielvis assumes RAS (xyz axis have higher numbers going right, anterior and superior)
% they relationship is simply:
% RAS = -(L)-(P)S
% 
% Additionally, iELvis and afni/suma are not in the same space
% afni/suma use the vox2RAS
% iElvis (freesurfer) the tkvox2RAS
% the offset between the two can be used to adjust the center (simple
% translation)
%
% I am still testing this: 
% not sure why the afni/suma and ielvis (freesurfer) are in different spaces
% AND
% not sure if it is always a simple translation or sometimes a full affine
% is required: using the full affine just to be safe
%--------------------------------------------------------------------------

%Eleonora September 2018

%% select location where the MRI of interest is:

% this NEEDS TO BE (should better be) the freesurfer folder that was used 
% to create the electrode locations that you are trying to convert!!
% eg if you have suma coords that beauchamp lab created, make sure that you
% are going to the fs/ folder for that subject on their server (or make a
% local copy of that on you fs/ folder on your computer)
% That is important as sometimes the average of 2 mris is used to align and
% perform the electrode localiz. and if you re-did the recon yourself using
% only 1 MRI then you might have some misalignments ... 

% You should point the folder with the imaing data that was used to do the  
% recon that was used to localize the electrodes you are trying to convert.
% this path is used to read out the header of the MRI that was used for the
% recon and get some information on the affine transformations needed
% between the original space (scanner space), the RAS space used by afni
% (which wants the electrodes in LPS for some reason) and the tk register
% space (which is used by freesurfer).

% e.g I copy over the files from beauchampserver to my local machine and
% then personalize them, so I point there. Other options are coded and only 
% partially tested (depends on how consistent the naming of the folders is)

directory_location = 'Beauchamplab_server';

switch directory_location
     case 'ECoG_server'
        % on ECoG_lab server: here there are the iELVIS related
        % recons and fs folder
        fsDir = '/Volumes/ecog/ECoG_Data/';
        subj_path=[subject 'Datafile/IMG/' subject];
     case 'Beauchamplab_server'
        % on Beauchamplab_server server here there are the afni/suma related
        % recons and fs folder
        fsDir = '/Volumes/data/UT/';
        subj_path=[subject '/iELVis_Localization/' subject];
    case 'eb_local'
        % on your local machine: this is /Applications/freesurfer/subjects/
        fsDir = getFsurfSubDir();
        subj_path=[subject '/fs'];
    case 'all_local'
        % on your local machine: this is /Applications/freesurfer/subjects/
        fsDir = getFsurfSubDir();
        subj_path=[subject];
    case 'StaticPattern'
        % on Yoshor_lab server (for Bill): same as Beauchamplab folder,
        % only copied over and personalized
        fsDir = '/Volumes/ecog/Yoshor_Lab/StaticPattern/';
        subj_path=[subject '/fs'];
end

subDir=fullfile(fsDir,subj_path);
if ~exist(subDir,'dir')
    error('Folder for %s is not present in FreeSurfer subjects directory (%s).',subj_path,fsDir);
end

%% load MRI header
mgzFname=fullfile(subDir,'mri','orig.mgz');
checkFile(mgzFname);
MRIhdr=MRIread(mgzFname,true);

%vox2RAS affine matrices:
Norig=MRIhdr.vox2ras; %vox2ras
Torig=MRIhdr.tkrvox2ras;

%% perform conversion:

% if the varargin is not empty and it has reverse as a value, transform FROM elvis/freesurfer TO afni/suma
% if the varargin is empty, default to transform from afni/suma to elvis/freesurfer
output_coords=[];

if ~isempty(varargin) && ismember(varargin,'reverse')
    % from coords in tk register space (freesurfer) to the "normal" space: 
    % convert and put in LPS space for afni/suma
    tkregister2afniRAS=(Norig*(Torig\[input_coords(:,1:3)';ones(size(input_coords,1),1)']))';
    %output_coords=tkregister2afniRAS(:,1:3);
    afniRAS2afniLPS=[-tkregister2afniRAS(:,1) -tkregister2afniRAS(:,2) tkregister2afniRAS(:,3)];
    output_coords=afniRAS2afniLPS;

else 
    % from coords in "normal" space to tkregister space (freesurfer): 
    % put in RAS space first and then convert
    afniLPS2afniRAS=[-input_coords(:,1) -input_coords(:,2) input_coords(:,3)];
    afniRAS2tkregister=(Torig*(Norig\[afniLPS2afniRAS(:,1:3)';ones(size(input_coords,1),1)']))';
    
    output_coords=afniRAS2tkregister(:,1:3);
end


% Norig-Toring
% center_shift = Norig(1:3,4)-Torig(1:3,4);
% this should be equal to the difference between the centers performed by
% the affine

end
